/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Connector for the PKI-API
 */


function PKIAPIConnector(cfg) {
	this.cfg = cfg;
}
PKIAPIConnector.constructor = PKIAPIConnector;

exports.PKIAPIConnector = PKIAPIConnector;



/**
 * Post an action to the PKI-API
 *
 * @param {Number} srID the Service Request ID
 * @param {Object} action the json action
 */
PKIAPIConnector.prototype.postAction = function(srID, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "postAction()");

	var url = this.cfg.apiURL + "/api/sr/" + srID + "/action";
	var c = new URLConnection(url);

	if (typeof(this.cfg.keyStore) != "undefined") {
		GPSystem.log(GPSystem.DEBUG, module.id, "Set trust store, key store and pw for PKI API connection");
		c.setTLSKeyStores(this.cfg.trustStore, this.cfg.keyStore, this.cfg.keyPassword);
	} else if (typeof(this.cfg.trustStore) != "undefined") {
		GPSystem.log(GPSystem.DEBUG, module.id, "Set only trust store for PKI API connection");
		c.setTLSKeyStores(this.cfg.trustStore);
	} else {
		GPSystem.log(GPSystem.DEBUG, module.id, "Didn't set tls key stores");
	}

	c.addHeaderField("Content-Type", "application/json");
	c.addHeaderField("Accept", "application/json");

	var json = JSON.stringify(action);
	GPSystem.log(GPSystem.DEBUG, module.id, "post action \n" + json + "\nto " + url);
	var rsp = c.post(json);
	GPSystem.log(GPSystem.DEBUG, module.id, "Response from PKI-API: " + rsp);

	var cookie = c.getHeaderField("Set-Cookie");

	if (!cookie) {
		GPSystem.log(GPSystem.DEBUG, module.id, "No session cookie returned by the PKI-API");
		return;
	}

	var combinedCookie = "";
	for (var i = 0; i < cookie.length; i++) {
		var eol = cookie[i].indexOf(";");
		if (eol < 0) {
			eol = cookie[i].length;
		}
		var elem = cookie[i].substr(0, eol);
		if (i > 0) {
			combinedCookie += "; ";
		}
		combinedCookie += elem;
	}

	var session = combinedCookie.substring("JSESSIONID=".length);
	GPSystem.log(GPSystem.DEBUG, module.id, "PKI-API session id: " + session);
	return session;
}
