/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2006 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A Bitmap
 */

/**
 * Create a bit map for the given number of bits.
 *
 * @class Class implementing a bit map
 * @constructor
 * @param {Number} size the number of bits in the bit map.
 */
function Bitmap(size) {
	this.size = size;
	this.bits = [];
	size = (size - 1 >> 3) + 1;
	for (var i = 0; i < size; i++) {
		this.bits.push(0);
	}
}

exports.Bitmap = Bitmap;



/**
 * Get the bit value at the zero-based position.
 *
 * @param {Number} pos the zero-based position in the bit map.
 * @type Number
 * @return 0 or 1
 */
Bitmap.prototype.get = function(pos) {
	if (pos < 0 || pos >= this.size) {
		throw new Error("pos out of range");
	}
	return (this.bits[pos >> 3] >> 7 - (pos & 0xF)) & 1;
}



/**
 * Toggle the bit at the zero-based position.
 *
 * @param {Number} pos the zero-based position in the bit map.
 */
Bitmap.prototype.toggle = function(pos) {
	if (pos < 0 || pos >= this.size) {
		throw new Error("pos out of range");
	}
	this.bits[pos >> 3] ^= 0x80 >> (pos & 0x7);
}



/**
 * Determine the position of the first unset bit.
 *
 * @type Number
 * @return the zero-based position of -1
 */
Bitmap.prototype.determineFirstUnset = function() {
	for (var i = 0; i < this.bits.length; i++) {
		var a = this.bits[i];
		if (a != 0xFF) {
			for (var j = 0; j < 8 && ((i << 3) + j < this.size); j++) {
				if ((a & 0x80) == 0) {
					return (i << 3) + j;
				}
				a <<= 1;
			}
		}
	}
	return -1;
}



Bitmap.test = function() {
	var map = new Bitmap(2);
	assert(map.determineFirstUnset() == 0);
	assert(map.get(0) == 0);
	map.toggle(0);
	assert(map.get(0) == 1);
	assert(map.determineFirstUnset() == 1);
	map.toggle(1);
	assert(map.determineFirstUnset() == -1);

	map.toggle(0);
	assert(map.get(0) == 0);
	assert(map.determineFirstUnset() == 0);

	var map = new Bitmap(256);

	for (var i = 0; i < 257; i++) {
		var id = map.determineFirstUnset();
		if (id == -1) {
			assert(i == 256);
			break;
		}
		map.toggle(id);
		print(map.bits);
	}

	for (var i = 0; i < 256; i++) {
		map.toggle(i);
		print(map.bits);
	}
}



