/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2025 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview X509HolderView
 */

var HolderView = require('pki-as-a-service/ui/HolderView').HolderView;
var PKIXCommon = require("scsh/x509/PKIXCommon").PKIXCommon;



function X509HolderView(service, holder) {
    this.service = service;
    this.holder = holder;
	GPSystem.log(GPSystem.DEBUG, module.id, "new X509HolderView()");
}

X509HolderView.prototype = Object.create(HolderView.prototype);
X509HolderView.constructor = X509HolderView;

exports.X509HolderView = X509HolderView;



X509HolderView.prototype.getDetails = function(homeURL, subjectController, subject) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var details = <details id="holderview"></details>;
	details.appendChild(this.getSummary(homeURL, subjectController, subject));

   	var certTable = this.renderCertificateTable(homeURL);
	if (certTable) {
       	var holderDAO = this.service.daof.getHolderDAO();
		var path = holderDAO.determineHolderPathList(this.holder);
		var hp = subjectController.renderHolderPath(path, homeURL);
		details.appendChild(hp);
		details.appendChild(certTable);
	}

	return details;
}



X509HolderView.prototype.renderCertificateTable = function(homeURL) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var certdao = this.service.daof.getCertificateDAO();

	var t = <table class="pure-table pure-table-horizontal"/>;
	t.thead += <thead>
			<tr>
				<th>Subject</th>
				<th>Issuer</th>
				<th>Serial</th>
				<th>Key Domain</th>
			</tr>
		</thead>;
	t.tbody += <tbody/>;

	var boList = certdao.enumerateCertificates(this.holder);

	if (boList.length == 0) {
		return t;
	}

	var signerDAO = this.service.daof.getSignerDAO();

	for (var i = 0; i < boList.length; i++) {
		var bo = boList[i];
		var cert = new X509(bo.bytes);
		var dn = cert.getSubjectDNString();
		try	{
			var rdn = PKIXCommon.parseDN(dn);
			var scn = PKIXCommon.findRDN(rdn, "CN");
		} catch(e) {
			scn = "*** Invalid DN ***";
		}

		if (!scn) {
			scn = dn;
		}

		var dn = cert.getIssuerDNString();
		try	{
			var rdn = PKIXCommon.parseDN(dn);
			var icn = PKIXCommon.findRDN(rdn, "CN");
		} catch(e) {
			icn = "*** Invalid DN ***";
		}

		if (!icn) {
			icn = dn;
		}

		var url = homeURL + "/cert/" + bo.id;

		var currentTag = "";
		if (boList.length > 1 && bo.id == this.holder.certId) {
			currentTag = String.fromCharCode(0xBB);
		}

		var signer = signerDAO.getSignerByKeyId(this.holder, bo.keyId);
		if (signer && signer.keyDomain) {
			var kdInfo = signer.keyDomain.toString(HEX);
			if (kdInfo.length > 16) {
				kdInfo = kdInfo.substring(0, 16) + "...";
			}
		} else {
			var kdInfo = "";
		}

		var tr =
		<tr>
			<td>
				{currentTag}
				<a href={ url }>{ scn }</a>
			</td>
			<td>{ icn }</td>
			<td>{ cert.getSerialNumberString() }</td>
			<td>{ kdInfo }</td>
		</tr>

		t.tbody.appendChild(tr);
	}

	return t;
}
