/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A view for service requests
 */

var CommonUI			= require('scsh/srv-cc1/CommonUI').CommonUI;
var Paginator			= require('pki-as-a-service/ui/Paginator').Paginator;
var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var ServiceRequestOverview	= require('pki-as-a-service/ui/ServiceRequestOverview').ServiceRequestOverview;



function ServiceRequestOverviewController(ui, isAuditView) {
	this.ui = ui;
	this.service = ui.service;
	this.isAuditView = isAuditView;

	this.paginator = new Paginator();

	this.filter = {};

	this.view = new ServiceRequestOverview(this);
}

exports.ServiceRequestOverviewController = ServiceRequestOverviewController;



ServiceRequestOverviewController.SELECT_ANY_VALUE = "Any";



ServiceRequestOverviewController.prototype.handleRequest = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleRequest()");
	GPSystem.log(GPSystem.DEBUG, module.id, "req.method:" + req.method);
	GPSystem.log(GPSystem.DEBUG, module.id, "req.queryString:" + req.queryString);

	if (req.queryString) {
		this.handleOperation(req, res);
	}

	if (!this.isAuditView && req.url.length != 2) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Invalid URL");
	}

	if (req.parameters) {
		this.handlePostOperation(req, res);
	}

	var page = this.getPage(req, res);
	this.ui.sendPage(req, res, req.url, page);
}



ServiceRequestOverviewController.prototype.handleOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleOperation(" + req.queryString + ")");
	var operation = CommonUI.parseQueryString(req.queryString);

	switch(operation.op) {
		case "changepagesize":
			var val = parseInt(operation.pagesize);
			if (val > 0) {

				this.paginator.pageSize = val;
			}
			if (val > this.paginator.offset) {
				this.paginator.offset = 0;
			}
			break;
		case "changeoffset":
			var val = parseInt(operation.offset);
			this.paginator.offset = val;
			break;
	}

	var url = this.ui.homeURL(req.url) + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}

	res.setStatus(HttpResponse.SC_SEE_OTHER);
	res.addHeader("Location", url);
}



ServiceRequestOverviewController.prototype.handlePostOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handlePostOperation( op=" + req.parameters.op + " )");
	if (req.parameters.op == "filter") {
		if (req.parameters.process) {
			this.updateProcessFilter(req.parameters.process);
		}

		if (typeof(req.parameters.details) != "undefined") {
			this.updateDetailsFilter(req.parameters.details);
		}

		if (req.parameters.lifeCycle) {
			this.updateLifeCycleFilter(req.parameters.lifeCycle);
		}
	}
}



ServiceRequestOverviewController.prototype.getPage = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = this.view.getOverview(req, res);

	var t = this.ui.generateTemplate(req.url);
 	var c = t..div.(@id == "content");
 	c.div = page;

	return t;
}



ServiceRequestOverviewController.prototype.getNumberOfServiceRequests = function() {
	var srDAO = this.service.daof.getServiceRequestDAO();

	if (this.isAuditView) {
		return srDAO.countServiceRequestsByFilter(this.filter);
	}

	return srDAO.countServiceRequests(this.ui.session.user.id, this.filter, this.ui.session.user.roles);
}



ServiceRequestOverviewController.prototype.getServiceRequests = function(records) {
	var srDAO = this.service.daof.getServiceRequestDAO();

	if (this.isAuditView) {
		GPSystem.log(GPSystem.DEBUG, module.id, "getServiceRequests as auditor");
		return srDAO.listServiceRequestsByFilter(this.filter, this.paginator.getOffset(records), this.paginator.pageSize);
	}
	GPSystem.log(GPSystem.DEBUG, module.id, "getServiceRequests for user");

	return srDAO.listServiceRequests(this.ui.session.user.id, this.ui.session.user.roles, this.filter, this.paginator.getOffset(records), this.paginator.pageSize);
}



ServiceRequestOverviewController.prototype.updateProcessFilter = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updateProcessFilter(process="  + process + ")");

	if (process == ServiceRequestOverviewController.SELECT_ANY_VALUE) {
		delete this.filter.process;
	} else {
		this.filter.process = process;
	}
}



ServiceRequestOverviewController.prototype.updateDetailsFilter = function(details) {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updateDetailsFilter(details=" + details + ")");

	this.view.detailsFilterValue = details;

	if (details.length == 0) {
		delete this.filter.details;
	} else {
		this.filter.details = {
			compOp: "like",
			value: "%" + details.toLowerCase() + "%"
		}
	}
}



ServiceRequestOverviewController.prototype.updateLifeCycleFilter = function(lifeCycle) {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updateLifeCycleFilter(lifeCycle=" + lifeCycle + ")");

	if (lifeCycle == ServiceRequestOverviewController.SELECT_ANY_VALUE) {
		delete this.filter.lifeCycle;
	} else {
		for (var i = 0; i < ServiceRequestModel.LIFECYCLE_STATES.length; i++) {
			var lc = ServiceRequestModel.LIFECYCLE_STATES[i];
			if (lifeCycle == lc) {
				this.filter.lifeCycle = i;
				break;
			}
		}
	}
}



ServiceRequestOverviewController.prototype.getProcesses = function() {
	var srDAO = this.service.daof.getServiceRequestDAO();
	return srDAO.getProcessList();
}



ServiceRequestOverviewController.prototype.getSubjectInfo = function(subjectId) {
	if (subjectId) {
		var subject = this.service.getSubject(subjectId);
		if (subject.getName()) {
			var value = subject.getName();
		} else {
			var value = subject.getEmail();
		}

		return value;
	}

	return "";
}
