/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2025 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A controller for the holder list view
 */

var CommonUI			= require('scsh/srv-cc1/CommonUI').CommonUI;
var Paginator			= require('pki-as-a-service/ui/Paginator').Paginator;
var HolderListView		= require('pki-as-a-service/ui/HolderListView').HolderListView;



function HolderListViewController(ui, isAuditView) {
	this.ui = ui;
	this.service = ui.service;
	this.isAuditView = isAuditView;
	this.paginator = new Paginator();
	this.view = new HolderListView(this);
}

exports.HolderListViewController = HolderListViewController;



HolderListViewController.prototype.handleRequest = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleRequest()");
	// GPSystem.log(GPSystem.DEBUG, module.id, "req.method:" + req.method);
	// GPSystem.log(GPSystem.DEBUG, module.id, "req.queryString:" + req.queryString);

	if (req.url.length < 3) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Invalid URL");
	}

	var holder = this.getHolder(req, res);
	if (!holder) {
		return;
	}

	if (!holder.subjectId) {
		// Authorization not possible
		GPSystem.log(GPSystem.DEBUG, module.id, "Authorization not possible because the holder is not assigned to a subjectId");
		this.ui.serveNotAuthorizedPage(req, res, req.url);
        return;
	}

	var subject = this.service.getSubject(holder.subjectId);
	if (!subject.canRead(this.ui.session.user) && !this.isAuditView) {
        //res.setStatus(HttpResponse.SC_FORBIDDEN);
        this.ui.serveNotAuthorizedPage(req, res, req.url);
        return;
    }

	if (req.queryString) {
		this.handleOperation(req, res);
	}

	var page = this.view.getPage(req, res, holder);
	this.ui.sendPage(req, res, req.url, page);
}



HolderListViewController.prototype.getHolder = function(req, res) {
	var id = parseInt(req.url[2]);
	if (typeof(id) != "number" || isNaN(id)) {
		res.setStatus(HttpResponse.SC_BAD_REQUEST);
		return;
	}

	var holderDAO = this.service.daof.getHolderDAO();
	var bo = holderDAO.getHolderById(id);
	if (!bo) {
		res.setStatus(HttpResponse.SC_NOT_FOUND);
        return;
	}

	return bo;
}



HolderListViewController.prototype.handleOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleOperation(" + req.queryString + ")");
	var operation = CommonUI.parseQueryString(req.queryString);

	switch(operation.op) {
		case "changepagesize":
			var val = parseInt(operation.pagesize);
			if (val > 0) {

				this.paginator.pageSize = val;
			}
			if (val > this.paginator.offset) {
				this.paginator.offset = 0;
			}
			break;
		case "changeoffset":
			var val = parseInt(operation.offset);
			this.paginator.offset = val;
			break;
	}

	var url = this.ui.homeURL(req.url) + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}

	res.setStatus(HttpResponse.SC_SEE_OTHER);
	res.addHeader("Location", url);
}



HolderListViewController.prototype.getHolderPathList = function(holder) {
    var holderDAO = this.service.daof.getHolderDAO();
    return holderDAO.determineHolderPathList(holder);
}



HolderListViewController.prototype.countSubHolder = function(holderId) {
	var holderDAO = this.service.daof.getHolderDAO();
	return holderDAO.countSubHolder(holderId);
}



HolderListViewController.prototype.getSubHolderList = function(holderId, records) {
    var holderDAO = this.service.daof.getHolderDAO();
    return holderDAO.getSubHolderList(holderId, this.paginator.getOffset(records), this.paginator.pageSize)
}



HolderListViewController.prototype.getSubjectInfo = function(subjectId) {
	if (subjectId) {
		var subject = this.service.getSubject(subjectId);
		if (subject.getName()) {
			var value = subject.getName();
		} else {
			var value = subject.getEmail();
		}

		return value;
	}

	return "";
}
