/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for CreateX509SignerServiceRequest
 */

var ServiceRequestModel = require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.xsigner.title": "Create X509 CA Signer",

		"msg.xsigner.cacertspec": "CA Certificate Definitions",

		"msg.xsigner.cadata": "Certification Authority Data",
		"msg.xsigner.caName": "Trust Center",
		"msg.xsigner.issuer": "Issuing CA",

		"msg.xsigner.rootCertValidityDays": "Expires after (days)",
		"msg.xsigner.caPathLen": "Subordinate CAs",
		"msg.xsigner.dn": "Distinguished Name",
		"msg.xsigner.rootKeySpecId": "Key Specification",
		"msg.xsigner.subKeySpecId": "Key Specification",
		"msg.xsigner.subCertValidityDays": "Expires after (days)",
		"msg.xsigner.dnmask": "Subject DN Rule",
		"msg.xsigner.supportedProfiles": "Supported Profiles",
		"msg.xsigner.crl": "CRL Definitions",
		"msg.xsigner.validityDaysCRL": "Expires after (days)",

		"msg.xsigner.keySpec": "Key Specification",

		"msg.xsigner.subjectspec": "Subject Certificate Definitions",


		"msg.xsigner.selectDomain": "Selected Key Domain",

		"msg.xsigner.seealso.signerToken": "Please connect a token where the signer should be created. {0}",
		"msg.xsigner.seealso.caToken": "Please connect and unlock your CA token. {0}",
		"msg.xsigner.seealso.pin": "Please unlock your connected device. {0}",

		"action.xsigner.create": "Create Signer",
		"action.xsigner.produce.hsmservice": "Issue Certificate",
		"action.xsigner.createcrl": "Create CRL"
	}}
]);



function CreateX509SignerRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new CreateX509SignerRequestController()");
}

exports.CreateX509SignerRequestController = CreateX509SignerRequestController;



CreateX509SignerRequestController.prototype.createRequest = function(srct, caSubjectId) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "CreateX509SignerServiceRequest",
		title: srct.i18n("msg.xsigner.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: caSubjectId,
		state: "Collecting CA information",
		content: {
			tokenId: srct.ui.session.user.tokenId
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



CreateX509SignerRequestController.prototype.postCreateAction = function(req, res, srct, action, trustCenter) {
	var bo = this.createRequest(srct, trustCenter.getId());

	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



CreateX509SignerRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.submit":
	case "action.save":
	case "action.cr.revoke":
	case "action.cr.updateStatus":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



CreateX509SignerRequestController.prototype.getCreateView = function(req, res, srct, trustCenter) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var subjectLink = srct.ui.homeURL(req.url) + "/subject/" + trustCenter.getId();

	var form =	<div>
				<h1>Create a new X509 CA Signer</h1>
				<p>This service will create a new certificate signer for the TrustCenter <a href={subjectLink}>{trustCenter.getName()}</a>.</p>
				<p></p>
				<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
				</form>
			</div>

	return form;
}



CreateX509SignerRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	if (sr.getLifeCycle() >= ServiceRequestModel.LIFECYCLE_INUSE) {
		var certView = srct.renderCertificateList(req, res, sr);
		if (certView) {
			div.appendChild( <br/> );
			div.appendChild( <h2>Certificates</h2> );
			div.appendChild( certView );
		}
	}

	return div;
}
