/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for sending a key to escrow
 */

var I18N = require('scsh/srv-cc1/I18N').I18N;
var BackendServiceRequestControllerBase = require('pki-as-a-service/ui/BackendServiceRequestControllerBase').BackendServiceRequestControllerBase;



I18N.addResources([
 	{ lang: "EN", keys: {
		"msg.kess.sendtoescrow": "Send Key to Escrow Data",
		"action.kess.enumeratekeys": "Enumerate Keys",
		"action.kess.sendtoescrow": "Send to Escrow",
 	}}
 ]);



function SendKeyToEscrowRequestController(plugin) {
	GPSystem.log(GPSystem.DEBUG, module.id, "new SendKeyToEscrowRequestController()");
	BackendServiceRequestControllerBase.call(this, plugin);
}

exports.SendKeyToEscrowRequestController = SendKeyToEscrowRequestController;

SendKeyToEscrowRequestController.prototype = Object.create(BackendServiceRequestControllerBase.prototype);



SendKeyToEscrowRequestController.prototype.postCreateAction = function(req, res, srct, action, subject, holder) {
	var recipientId = this.validateHolder(srct, "KeyEscrow", req.parameters);
	var originatorId = subject != undefined ? subject.getId() : srct.ui.session.user.id;
	// Validity already checked in validateHolder()
	var keyEscrowHolderId = parseInt(req.parameters.keyescrow);

	var data = { process: "SendKeyToEscrow",
		originatorId: originatorId,
		recipientId: recipientId,
		content: {
			keyEscrowHolderId: keyEscrowHolderId
		}
	};

	return BackendServiceRequestControllerBase.prototype.postCreateActionViaAPI.call(this, req, res, srct, data);
}



SendKeyToEscrowRequestController.prototype.isHolderCandidate = function(srct, holder) {
	if (typeof(this.groupSignerHolderId) == "undefined") {
		return true;
	}

	var gsid = holder.getContent().groupSignerHolderId;
	return this.groupSignerHolderId == gsid;
}



SendKeyToEscrowRequestController.prototype.getCreateView = function(req, res, srct, subject, holder) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page =	<div>
				<h1>Send Key to Escrow</h1>

				<p>Key escrow is a secure storage of cryptographic keys and their associated metadata outside the
				   secure confines of a SmartCard-HSM. It acts as a secure key backup and key distribution mechanism.</p>

				<p>Key escrow must be established by creating a key domain and an escrow key within that domain.
				   The escrow key represents the trusted entity for which cryptographic keys
				   are encrypted. Keys can only migrate within the key domain.</p>

				<p>This service request allows you to select a SmartCard-HSM, either one assigned to you or
				   a SmartCard-HSM of an entity you control. The service request will enumerate cryptographic
				   keys that are in the same key domain as the escrow key and that are not already held in
				   escrow. You can select multiple keys to be included in the transaction.</p>

				<p>The service request will then create an export key on the SmartCard-HSM, derive an Exchange Key Encryption Key (XKEK)
				   with the escrow key as target and wrap the selected keys. The wrapped keys are store in the database.</p>

			</div>

	if (holder.type == "KeyEscrow") {
		var form =
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
				<input type="hidden" id="keyescrow" name="keyescrow" value={ holder.id } />
				<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		page.appendChild(form);
		return page;
	}

	// Filter escrow keys in the same group
	this.groupSignerHolderId = holder.parentId;

	var select = this.getHolderSelect(srct, "KeyEscrow" );

	if (select == undefined) {
		page.appendChild(
			<p>&#9888; There is no Key Escrow available.</p>
		);
		return page;
	}

	var form =
		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<p>Please select a Key Escrow to which you would like to send keys.</p>
		</form>

	form.appendChild(select);
	form.appendChild(
		<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
		 );

	page.appendChild(form);

	return page;
}
