/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2025 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A holder list view
 */

function HolderListView(controller) {
	this.controller = controller;
	this.paginator = controller.paginator;
	this.ui = controller.ui;
}

exports.HolderListView = HolderListView;



HolderListView.prototype.getPaginationDiv = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var pagination =
		<div class="table-control">
			<div class="">
				<form class="pure-form" action="" method="get" onchange="submit()">
					<input name="op" type="hidden" value="changepagesize"/>
					<fieldset>
						<label for="size">Table Size:</label>
						<select id="size" name="pagesize"/>
						<noscript><button class="pure-button" type="submit">Change</button></noscript>
					</fieldset>
				</form>
			</div>
		</div>;

	var selectSize = pagination..select.(@id == "size");

	var option = <option>5</option>;
	if (this.paginator.pageSize == 5) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>10</option>;
	if (this.paginator.pageSize == 10) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>20</option>;
	if (this.paginator.pageSize == 20) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>50</option>;
	if (this.paginator.pageSize == 50) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>100</option>;
	if (this.paginator.pageSize == 100) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	return pagination;
}



HolderListView.prototype.getSubjectLink = function(url, holder) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	if (holder.subjectId) {
		var value = this.controller.getSubjectInfo(holder.subjectId);
		return <a href={url + "/subject/" + holder.subjectId + "#" + holder.id}>{value}</a>;
	}

	return "";
}



HolderListView.prototype.renderHolderList = function(req, res, parent) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div class="card">
		<h2>Children</h2>
	</div>;

	// Create page navigation

	var pagination = this.getPaginationDiv();

	var count = this.controller.countSubHolder(parent.id);

	var homeURL = this.ui.homeURL(req.url);
	var url = homeURL + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}

	var pagenavigator = this.paginator.getPagination(count, url + "?op=changeoffset&");

	// Create table
	var holderList = this.controller.getSubHolderList(parent.id, count);
	var table = this.renderHolderTable(holderList, homeURL);

	var hp = this.renderHolderPath(parent, homeURL);
	div.appendChild(hp);

	var tableDiv = <div class="pure-u-1">{table}</div>;
	pagination.appendChild(tableDiv);
	pagination.appendChild(pagenavigator);
	div.appendChild(pagination);
	return div;
}



HolderListView.prototype.renderHolderTable = function(holderList, homeURL) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var table =
		<table class="pure-table pure-table-striped" style="table-layout:fixed">
			<thead>
				<tr>
					<th>Type</th>
					<th>Name</th>
					<th>Subject</th>
					<th>Signer No</th>
					<th>Certificate</th>
				</tr>
			</thead>
			<tbody/>
		</table>;

	for (var i = 0; i < holderList.length; i++) {
		var holder = holderList[i];

		if (holder.type) {
			var type = holder.type;
		} else {
			var type = "";
		}

		if (holder.subjectId) {
			var name = <a href={homeURL + "/holder/" + holder.id}>{holder.name}</a>;
		} else {
			name = holder.name;
		}


		var subject = this.getSubjectLink(homeURL, holder);

		if (holder.certId) {
			var certLink = <a href={homeURL + "/cert/" + holder.certId}>{holder.certId}</a>;
		} else {
			var certLink = "";
		}

		var tr = <tr>
				<td>{type}</td>
				<td>{name}</td>
				<td>{subject}</td>
				<td>{holder.signerNo}</td>
				<td>{certLink}</td>
			</tr>

 		table.tbody.appendChild(tr);
	}

	return table;
}



HolderListView.prototype.renderHolderPath = function(holder, homeURL) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var path = this.controller.getHolderPathList(holder);

	var h = <h3 id={path[path.length - 1].id}/>;

	for (var i = 0; i < path.length; i++) {
		var holder = path[i];
		if (i > 0) {
			h.appendChild('/');
		}
		if (i < path.length - 1) {
			if (holder.subjectId) {
				h.appendChild(<a href={homeURL + "/holder/" + holder.id}>{holder.name}</a>);
			} else {
				h.appendChild(<span>{holder.name}</span>);
			}
		} else {
			h.appendChild(holder.name);
		}
	}

	return h;
}



HolderListView.prototype.getOverview = function(req, res, holder) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

    var title = "Holder";

	var homeURL = this.ui.homeURL(req.url);
	var subjectLink = this.getSubjectLink(homeURL, holder);

	if (holder.subjectId) {
		var subject = this.ui.service.getSubject(holder.subjectId);
		var holderView = this.ui.subjectController.renderHolderView(holder, subject, homeURL);
	}

	var page =	<div>
				<h1>{title}</h1>
				<p class="fromto">{subjectLink}</p>
				{holderView}

				<br/>
				<div id="list" />
				<br/>
			</div>;

	var div = page..div.(@id == "list");

	var overview = this.renderHolderList(req, res, holder);
	div.appendChild(overview);

	return page;
}



HolderListView.prototype.getPage = function(req, res, holder) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = this.getOverview(req, res, holder);

	var t = this.ui.generateTemplate(req.url);
 	var c = t..div.(@id == "content");
 	c.div = page;

	return t;
}
