/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2018 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service Request model for PKCS#10 certificate requests
 */



var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var CertificateRequestModel	= require('pki-as-a-service/processes/CertificateRequestModel').CertificateRequestModel;
var PKCS10			= require('scsh/pkcs/PKCS10').PKCS10;
var X509Name			= require('scsh/x509/X509Name').X509Name;
var PKIXCommon			= require('scsh/x509/PKIXCommon').PKIXCommon;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function PKCS10RequestModel(service, bo) {
	CertificateRequestModel.call(this, service, bo);
	this.validationResult = [];
}

PKCS10RequestModel.prototype = Object.create(CertificateRequestModel.prototype);
PKCS10RequestModel.constructor = PKCS10RequestModel;

exports.PKCS10RequestModel = PKCS10RequestModel;



PKCS10RequestModel.prototype.getForm = function(user) {
	if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT) {
		if (this.form == undefined) {
			this.form = [{
				id: "applicant",
				legend: "msg.cr.applicant",
				fields: [
					{ id: "creq", label: "msg.pkcs10.req", type: "file", required: this.model.creq ? false : true, editable: true, value: this.model.creq ? "uploaded-csr" : "nothing" }
				]}
			];

			if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
				var req = this.getRequest();
				var subject = req.getSubject();
				var cn = "";
				if (subject.elements > 0) {
					var xName = new X509Name(subject)
					var reqDN = xName.toString();

					try {
						cn = xName.getRDNAsString(X509Name.commonName);
					} catch(e) {
						GPSystem.log(GPSystem.DEBUG, module.id, e);
					}
				}
				var key = req.getPublicKey();
				var keySize = key.getSize();
				if (key.getComponent(Key.MODULUS)) {
					var keyType = "RSA";
				} else {
					var keyType = "EC";
					var curve = ASN1.nameForObjectIdentifier(key.getComponent(Key.ECC_CURVE_OID));
				}

				if (reqDN) {
					this.form[0].fields.push({ id: "reqDN", label: "msg.cr.dn", type: "text", size: 64, required: false, editable: false, value: reqDN });
				}

				this.form[0].fields.push({ id: "commonName", label: "msg.cr.commonName", type: "text", size: 64, required: true, editable: true, value: (this.model.commonName ? this.model.commonName : cn) });

				this.form[0].fields.push({ id: "keyType", label: "msg.cr.keyType", type: "text", size: 64, required: false, editable: false, value: keyType });

				this.form[0].fields.push({ id: "keySize", label: "msg.cr.keySize", type: "text", size: 64, required: false, editable: false, value: keySize });
				if (curve) {
					this.form[0].fields.push({ id: "curve", label: "msg.cr.curve", type: "text", size: 64, required: false, editable: false, value: curve });
				}
			}
		}
	} else {
		this.form = CertificateRequestModel.prototype.getForm.call(this, user)
	}
	return this.form;
}



PKCS10RequestModel.prototype.checkPKCS10Request = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "checkPKCS10Request()");

	if (this.model.creq && val.length == 0) {
		// Allow empty submission
		// if the request was already uploaded
		return true;
	}

	try {
		if (val.byteAt(0) != 0x30) {
			var pem = val.toString(ASCII);
			GPSystem.log(GPSystem.DEBUG, module.id, "Converting PEM " + pem);
			var val = PKIXCommon.parsePEM("CERTIFICATE REQUEST", pem);

			if (val == null || val.length == 0) {
				var val = PKIXCommon.parsePEM("NEW CERTIFICATE REQUEST", pem);
			}
			if (val == null || val.length == 0) {
				var msg = "Could not find a certificate request headed in the uploaded PEM";
				GPSystem.log(GPSystem.INFO, module.id, msg);
				fld.message = msg;
				return false;
			}
		}
		var req = new PKCS10(val);
		if (!req.verify()) {
			var msg = "Verification of the uploaded PKCS10 request failed";
			GPSystem.log(GPSystem.INFO, module.id, msg);
			fld.message = msg;
			return false;
		}
		this.model[fld.id] = val;
	}
	catch(e) {
		GPSystem.log(GPSystem.ERROR, module.id, e.message);
		fld.message = e.message;
		return false;
	}
	return true;
}



PKCS10RequestModel.prototype.updateModel = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "updateModel(" + fld.id + "," + val + ")");

	switch(fld.id) {
		case "creq":
			return this.checkPKCS10Request(fld, val);
	}
	return CertificateRequestModel.prototype.updateModel.call(this, fld, val);
}



PKCS10RequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];


	var raOfficer = this.isAssignedToRegistrationOfficer(user);

	if ((user.id == this.getOriginatorId()) || raOfficer) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) {
			this.actionList.push("action.pkcs10.upload");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			this.actionList.push("action.save");
			this.actionList.push("action.submit");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_DELIVER) {
			this.actionList.push("action.pkcs10.accept");
			this.appendRevocationActions(user, this.actionList);
		}
	}

	if (this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_SUBMIT
		&& this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_DELIVER
	) {
		this.actionList = undefined;
		this.actionList = CertificateRequestModel.prototype.getActionList.call(this, user);
		GPSystem.log(GPSystem.DEBUG, module.id, "actionlist=" + this.actionList);
	}

	return this.actionList;
}



PKCS10RequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.pkcs10.upload":
			this.handleUpload();
			break;
		case "action.submit":
			this.submitRequest();
			break;
		case "action.pkcs10.accept":
			this.activateCertificate();
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
			this.setStatusInfo("Certificate accepted");
			break;
		default:
			return CertificateRequestModel.prototype.perform.call(this, user, action);
	}

	this.commit(user.id);
	return true;
}



PKCS10RequestModel.prototype.getRequest = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "getRequest()");

	if (this.model.creq.byteAt(0) != 0x30) {
		var pem = this.model.creq.toString(ASCII);
		GPSystem.log(GPSystem.DEBUG, module.id, "Converting PEM " + pem);
		var req = PKIXCommon.parsePEM("CERTIFICATE REQUEST", pem);
		return new PKCS10(req);
	}

	GPSystem.log(GPSystem.DEBUG, module.id, this.model.creq);
	return new PKCS10(this.model.creq);
}



PKCS10RequestModel.prototype.handleUpload = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleUpload()");
	var req = this.getRequest();
	var subject = req.getSubject();
	var xName = new X509Name(subject)
	try {
		this.model.commonName = xName.getRDNAsString(X509Name.commonName);
	} catch(e) {
		GPSystem.log(GPSystem.DEBUG, module.id, e);
	}
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
}



PKCS10RequestModel.prototype.submitRequest = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "submitRequest()");

	this.getCertificateHolder();

	this.toApproval();
}
