/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for ImportX509SignerServiceRequest
 */

var ServiceRequestModel = require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var ImportSignerWizard = require('pki-as-a-service/ui/ImportSignerWizard').ImportSignerWizard;
var CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
var SubjectDAO = require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
var RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.impsigner.title": "Import X509 CA Signer",

		"msg.impsigner.subjectspec": "Subject Certificate Definitions",
		"msg.impsigner.subKeySpecId": "Key Specification",
		"msg.impsigner.subCertValidityDays": "Expires after (days)",
		"msg.impsigner.dnmask": "Subject DN Rule",
		"msg.impsigner.supportedProfiles": "Supported Profiles",

		"action.impsigner.import": "Import Signer",

		"msg.impsigner.caPathLen": "Subordinate CAs",
		"msg.impsigner.crl": "CRL Definitions",
		"msg.impsigner.validityDaysCRL": "Expires after (days)"
// 		"action.xsigner.produce.hsmservice": "Issue Certificate",
// 		"action.xsigner.createcrl": "Create CRL"
	}}
//	,
//	{ lang: "DE", keys: {
//		"action.save": "Speichern"
//	}}
]);



function ImportX509SignerRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new ImportX509SignerRequestController()");
}

exports.ImportX509SignerRequestController = ImportX509SignerRequestController;



ImportX509SignerRequestController.prototype.createRequest = function(srct, trustCenterId) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "ImportX509SignerServiceRequest",
		title: srct.i18n("msg.impsigner.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: trustCenterId,
		state: "Created",
		content: {
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}




ImportX509SignerRequestController.prototype.postCreateAction = function(req, res, srct, action, trustCenter) {
	var bo = this.createRequest(srct, trustCenter.getId());

	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



ImportX509SignerRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.submit":
	case "action.save":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	case "action.impsigner.import":
		srct.ui.wizard = new ImportSignerWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	case "action.impsigner.enumerate":
		srct.ui.wizard = new CardActionWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



ImportX509SignerRequestController.prototype.getCreateView = function(req, res, srct, trustCenter) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var subjectLink = srct.ui.homeURL(req.url) + "/subject/" + trustCenter.getId();

	var form =	<div>
				<h1>Import X509 Signer</h1>
				<p>This service request allows to import an existing X509 signer on a SmartCard-HSM for
				   use with the trustcenter <a href={subjectLink}>{trustCenter.getName()}</a>.</p>
				<p>The signer private key and CA certificate must be stored in the SmartCard-HSM and the
				   device must present at the workstation during import. For later use it must be connected
				   to the portal.</p>
				<p>This service request is usefull to import a CA that already exists outside the portal.
				   To prepare the SmartCard-HSM and import the CA key and certificate see the
				   scrips in the SDK or the import function in the Smart Card Shell.</p>
				<p>The service request will query parameter used by the CA to issue certificates, but in
				   contrast to the Create Signer service request it will not create a fresh key and
				   obtain a self-signer certificate or a certificate from a superior CA.</p>
				<p>For the import to be successful, an issuing CA certificate must be present in the system.
				   This can be either a different CA or a certificate imported as trusted.</p>
				<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
				</form>
			</div>

	return form;
}



ImportX509SignerRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	if (sr.getLifeCycle() >= ServiceRequestModel.LIFECYCLE_INUSE) {
		var certView = srct.renderCertificateList(req, res, sr);
		if (certView) {
			div.appendChild( <br/> );
			div.appendChild( <h2>Certificates</h2> );
			div.appendChild( certView );
		}
	}

	return div;
}
