/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2025 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A controller for the token list view
 */

var CommonUI			= require('scsh/srv-cc1/CommonUI').CommonUI;
var RoleDAO				= require('scsh/pki-db/db/RoleDAO').RoleDAO;
var Paginator			= require('pki-as-a-service/ui/Paginator').Paginator;
var TokenListView		= require('pki-as-a-service/ui/TokenListView').TokenListView;



function TokenListViewController(ui, isAuditView) {
	this.ui = ui;
	this.service = ui.service;
	this.isAuditView = isAuditView;

	this.paginator = new Paginator();

	this.filter = {};

	this.view = new TokenListView(this);
}

exports.TokenListViewController = TokenListViewController;



TokenListViewController.prototype.handleRequest = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleRequest()");
	GPSystem.log(GPSystem.DEBUG, module.id, "req.method:" + req.method);
	GPSystem.log(GPSystem.DEBUG, module.id, "req.queryString:" + req.queryString);

	if (req.queryString) {
		this.handleOperation(req, res);
	}

	if (!this.isAuditView && req.url.length != 2) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Invalid URL");
	}

	if (req.parameters) {
		this.handlePostOperation(req, res);
	}

	var page = this.getPage(req, res);
	this.ui.sendPage(req, res, req.url, page);
}



TokenListViewController.prototype.handleOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleOperation(" + req.queryString + ")");
	var operation = CommonUI.parseQueryString(req.queryString);

	switch(operation.op) {
		case "changepagesize":
			var val = parseInt(operation.pagesize);
			if (val > 0) {

				this.paginator.pageSize = val;
			}
			if (val > this.paginator.offset) {
				this.paginator.offset = 0;
			}
			break;
		case "changeoffset":
			var val = parseInt(operation.offset);
			this.paginator.offset = val;
			break;
	}

	var url = this.ui.homeURL(req.url) + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}

	res.setStatus(HttpResponse.SC_SEE_OTHER);
	res.addHeader("Location", url);
}



TokenListViewController.prototype.handlePostOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handlePostOperation( op=" + req.parameters.op + " )");
	if (req.parameters.op == "filter") {
		if (typeof(req.parameters.path) != "undefined") {
			this.updatePathFilter(req.parameters.path);
		}
	}
}



TokenListViewController.prototype.updatePathFilter = function(path) {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updatePathFilter(path="  + path + ")");

	this.view.pathFilterValue = path;

	if (path.length == 0) {
		delete this.filter.path;
	} else {
		this.filter.path = {
			compOp: "like",
			value: "%" + path.toLowerCase() + "%"
		}
	}
}



TokenListViewController.prototype.getPage = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = this.view.getOverview(req, res);

	var t = this.ui.generateTemplate(req.url);
 	var c = t..div.(@id == "content");
 	c.div = page;

	return t;
}



TokenListViewController.prototype.getNumberOfToken = function() {
	var tokenDAO = this.service.daof.getTokenDAO();

	if (this.isTokenManager()) {
		GPSystem.log(GPSystem.DEBUG, module.id, "getNumberOfToken as TokenManager");
		return tokenDAO.countTokenList(this.filter, this.paginator.getOffset(this.filter), this.paginator.pageSize);
	}

	return tokenDAO.countTokenByManagedByRoleId(this.ui.session.user.roles, this.filter);
}



TokenListViewController.prototype.getTokenActions = function(tokenId) {
	var tokenDAO = this.service.daof.getTokenDAO();
	var tokenActions = tokenDAO.enumerateTokenAction(tokenId);
	return tokenActions;
}



TokenListViewController.prototype.isTokenManager = function(records) {
	return this.ui.session.user.roles.indexOf(RoleDAO.ID_TokenManager) >= 0;
}



TokenListViewController.prototype.getTokenList = function(records) {
	var tokenDAO = this.service.daof.getTokenDAO();

	if (this.isTokenManager()) {
		GPSystem.log(GPSystem.DEBUG, module.id, "getTokenList as TokenManager");
		return tokenDAO.getTokenList(this.filter, this.paginator.getOffset(records), this.paginator.pageSize);
	}

	return tokenDAO.getTokenListByManagedByRoleId(this.ui.session.user.roles, this.filter, this.paginator.getOffset(records), this.paginator.pageSize);
}



TokenListViewController.prototype.getSubjectInfo = function(subjectId) {
	if (subjectId) {
		var subject = this.service.getSubject(subjectId);
		if (subject.getName()) {
			var value = subject.getName();
		} else {
			var value = subject.getEmail();
		}

		return value;
	}

	return "";
}
