/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2025 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview CVCHolderView
 */

var HolderView = require('pki-as-a-service/ui/HolderView').HolderView;
var CVC = require('scsh/eac/CVC').CVC;



function CVCHolderView(service, holder) {
    this.service = service;
    this.holder = holder;
	GPSystem.log(GPSystem.DEBUG, module.id, "new CVCHolderView()");
}

CVCHolderView.prototype = Object.create(HolderView.prototype);
CVCHolderView.constructor = CVCHolderView;

exports.CVCHolderView = CVCHolderView;



CVCHolderView.prototype.getDetails = function(homeURL, subjectController, subject) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var details = <details id="holderview"></details>;
	details.appendChild(this.getSummary(homeURL, subjectController, subject));

   	var certTable = this.renderCertificateTable(homeURL);
	if (certTable) {
       	var holderDAO = this.service.daof.getHolderDAO();
		var path = holderDAO.determineHolderPathList(this.holder);
		var hp = subjectController.renderHolderPath(path, homeURL);
		details.appendChild(hp);
		details.appendChild(certTable);
	}

	return details;
}



CVCHolderView.prototype.getCertificates = function() {
	var certDAO = this.service.daof.getCertificateDAO();
	var certs = certDAO.enumerateCertificates(this.holder);
	return certs;
}



CVCHolderView.prototype.getSigner = function(cert) {
	var signerDAO = this.service.daof.getSignerDAO();
	var signer = signerDAO.getSignerByKeyId(this.holder, cert.keyId);

	return signer;
}



CVCHolderView.prototype.renderCertificateTable = function(homeURL) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var certs = this.getCertificates();
	if (certs.length == 0) {
		return <p>No Signer was created for this holder.</p>
	}
	var t = <table class="pure-table pure-table-horizontal"/>;
	t.thead += <thead>
		<tr>
			<th>CAR</th>
			<th>CHR</th>
			<th>Key Domain</th>
		</tr>
	</thead>;
	t.tbody += <tbody/>;

	var holderDAO = this.service.daof.getHolderDAO();
	for (var i = 0; i < certs.length; i++) {
		var cert = certs[i];

		var cvc = new CVC(cert.bytes);

		if (this.holder.parentId) {
			var parent = holderDAO.getHolderById(this.holder.parentId);
			var url = homeURL + "/cert/" + parent.certId;
			var issuer = <a href={ url }>{ cvc.getCAR() }</a>;
		} else {
			var issuer = cvc.getCAR();
		}

		var url = homeURL + "/cert/" + cert.id;
		var signer = this.getSigner(cert);

		var kdInfo = "No Signer";
		if (signer) {
			kdInfo = signer.keyDomain.toString(HEX);
			if (kdInfo.length > 16) {
				kdInfo = kdInfo.substring(0, 16) + "...";
			}
		}

		var mark = cert.id == this.holder.certId ? "*" : "";

		var tr = <tr>
			<td>{ issuer }</td>
			<td><a href={ url }>{ cvc.getCHR() }</a>{ mark }</td>
			<td>{ kdInfo }</td>
		</tr>;

		t.tbody.appendChild(tr);
	}

	return t;
}
