/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2025 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A token list view
 */

var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;



function TokenListView(controller) {
	this.controller = controller;
	this.paginator = controller.paginator;
	this.ui = controller.ui;
}

exports.TokenListView = TokenListView;



TokenListView.prototype.getPaginationDiv = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var pagination =
		<div class="table-control">
			<div class="">
				<form class="pure-form" action="" method="get" onchange="submit()">
					<input name="op" type="hidden" value="changepagesize"/>
					<fieldset>
						<label for="size">Table Size:</label>
						<select id="size" name="pagesize"/>
						<noscript><button class="pure-button" type="submit">Change</button></noscript>
					</fieldset>
				</form>
			</div>
		</div>;

	var selectSize = pagination..select.(@id == "size");

	var option = <option>5</option>;
	if (this.paginator.pageSize == 5) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>10</option>;
	if (this.paginator.pageSize == 10) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>20</option>;
	if (this.paginator.pageSize == 20) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>50</option>;
	if (this.paginator.pageSize == 50) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>100</option>;
	if (this.paginator.pageSize == 100) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	return pagination;
}



TokenListView.prototype.getSubjectLink = function(url, subjectId) {
	if (subjectId) {
		var value = this.controller.getSubjectInfo(subjectId);
		return <a href={url + "/subject/" + subjectId}>{value}</a>;
	}

	return "";
}



TokenListView.prototype.getTypeString = function(type) {
	var str = "";

	if (!type) {
		return str;
	}

	var tb = ByteString.valueOf(type);
	GPSystem.log(GPSystem.DEBUG, module.id, "Token type " + tb);

	if (tb.byteAt(0) == 1) {
		str += "SC-HSM ";
	}

	str += "Version " + tb.byteAt(2) + "." + tb.byteAt(3);

	switch (tb.byteAt(1)) {
		case 0: str += " on JCOP"; break;
		case 1: str += " Demo on JCOP"; break;
		case 2: str += " on JCOP 2.4.1r3"; break;
		case 3: str += " on JCOP 2.4.2r3"; break;
		case 4: str += " on JCOP 2.4.2r1"; break;
		case 5: str += " on JCOP 3"; break;
		case 6: str += " on JCOP 4"; break;
		case 7: str += " on JCOP 4.5"; break;
	}

	return str;
}



TokenListView.prototype.renderTokenList = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div class="card"/>;

	// Create page navigation

	var pagination = this.getPaginationDiv();

	var count = this.controller.getNumberOfToken();

	var homeURL = this.ui.homeURL(req.url);
	var url = homeURL + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}

	var pagenavigator = this.paginator.getPagination(count, url + "?op=changeoffset&");

	// Create table header

	var table =
		<table class="pure-table pure-table-striped" style="table-layout:fixed">
			<thead>
				<tr>
					<th>Type</th>
					<th>Hardware</th>
					<th>Path</th>
					<th>Last Seen</th>
					<th>Subject</th>
					<th>Token Actions</th>
				</tr>
				<tr>
					<th></th>
					<th></th>
					<th></th>
					<th></th>
					<th></th>
					<th></th>
				</tr>
			</thead>
			<tbody/>
		</table>;

	var url = this.ui.homeURL(req.url);

	// Fill table


	var tokenList = this.controller.getTokenList(count);

	for (var i = 0; i < tokenList.length; i++) {
		var token = tokenList[i];

		var subjectId = this.getSubjectLink(url, token.subjectId);

        if (typeof(token.hardware) != "undefined") {
            var hardware = token.hardware;
        } else {
            hardware = "";
        }

        var tokenActions = this.controller.getTokenActions(token.id);
        var taDiv = <div/>
        for (var j = 0; j < tokenActions.length; j++) {
            var ta = tokenActions[j];
            taDiv.appendChild(<a href={url + "/sr/" + ta.serviceRequestId}>{ta.id}</a>);
        }

		var tr = <tr>
				<td>{this.getTypeString(token.type)}</td>
				<td>{hardware}</td>
				<td><a href={url + "/sr/" + token.serviceRequestId}>{token.path}</a></td>
				<td>{token.lastSeen}</td>
				<td>{subjectId}</td>
				<td>{taDiv}</td>
			</tr>
 		table.tbody.appendChild(tr);
	}

	var tableDiv = <div class="pure-u-1">{table}</div>;
	pagination.appendChild(tableDiv);
	pagination.appendChild(pagenavigator);
	div.appendChild(pagination);
	return div;
}



TokenListView.prototype.getOverview = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

    var title = "Token List";
	if (this.controller.isTokenManager()) {
		title += " (TokenManager View)";
	}

	var page =	<div>
				<h1>{title}</h1>

				<br/>
				<div id="list" />
				<br/>
			</div>;

	var div = page..div.(@id == "list");

	var overview = this.renderTokenList(req, res);
	div.appendChild(overview);

	return page;
}
