/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A view for service requests
 */

var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;



function ServiceRequestOverview(controller) {
	this.controller = controller;
	this.paginator = controller.paginator;
	this.filter = controller.filter;
	this.ui = controller.ui;

	this.detailsFilterValue = "";
}

exports.ServiceRequestOverview = ServiceRequestOverview;



ServiceRequestOverview.SELECT_ANY_VALUE = "Any";



ServiceRequestOverview.prototype.getProcessFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var processFilter = <form class="pure-form" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<input name="op" type="hidden" value="filter"/>
		<select id="processfilter" name="process" class="pure-u-23-24"/>
		<noscript><button class="pure-button" type="submit">Filter</button></noscript>
	</form>

	var selectProcess = processFilter..select.(@id == "processfilter");

	var option = <option>{ServiceRequestOverview.SELECT_ANY_VALUE}</option>;
	if (this.filter.process == ServiceRequestOverview.SELECT_ANY_VALUE) {
		option.@selected = "selected";
	}
	selectProcess.appendChild(option);

	var processes = this.controller.getProcesses();
	for (var i = 0; i < processes.length; i++) {
		var option = <option>{processes[i]}</option>;
		if (this.filter.process == processes[i]) {
			option.@selected = "selected";
		}
		selectProcess.appendChild(option);
	}

	return processFilter;
}



ServiceRequestOverview.prototype.getDetailsFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";


	var detailsFilter = <form class="pure-form" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<input name="op" type="hidden" value="filter"/>
		<input id="detailsFilter" type="search" placeholder="Search for Details" class="pure-u-23-24" name="details" value={this.detailsFilterValue}/>
		<noscript><button class="pure-button" type="submit">Filter</button></noscript>
	</form>

	return detailsFilter;
}



ServiceRequestOverview.prototype.getLifeCycleFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var lifeCycleFilter = <form class="pure-form" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<input name="op" type="hidden" value="filter"/>
		<select id="lifeCycleFilter" name="lifeCycle" class="pure-u-23-24"/>
		<noscript><button class="pure-button" type="submit">Filter</button></noscript>
	</form>

	var selectLifeCycle = lifeCycleFilter..select.(@id == "lifeCycleFilter");

	var option = <option>{ServiceRequestOverview.SELECT_ANY_VALUE}</option>;
	if (this.filter.lifeCycle == ServiceRequestOverview.SELECT_ANY_VALUE) {
		option.@selected = "selected";
	}
	selectLifeCycle.appendChild(option);

	for (var i = 0; i < ServiceRequestModel.LIFECYCLE_STATES.length; i++) {
		var lifeCycle = ServiceRequestModel.LIFECYCLE_STATES[i];
		var option = <option>{lifeCycle}</option>;
		if (this.filter.lifeCycle == i) {
			option.@selected = "selected";
		}
		selectLifeCycle.appendChild(option);
	}

	return lifeCycleFilter;
}



ServiceRequestOverview.prototype.getPaginationDiv = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var pagination =
		<div class="table-control">
			<div class="">
				<form class="pure-form" action="" method="get" onchange="submit()">
					<input name="op" type="hidden" value="changepagesize"/>
					<fieldset>
						<label for="size">Table Size:</label>
						<select id="size" name="pagesize"/>
						<noscript><button class="pure-button" type="submit">Change</button></noscript>
					</fieldset>
				</form>
			</div>
		</div>;

	var selectSize = pagination..select.(@id == "size");

	var option = <option>5</option>;
	if (this.paginator.pageSize == 5) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>10</option>;
	if (this.paginator.pageSize == 10) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>20</option>;
	if (this.paginator.pageSize == 20) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>50</option>;
	if (this.paginator.pageSize == 50) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>100</option>;
	if (this.paginator.pageSize == 100) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	return pagination;
}



ServiceRequestOverview.prototype.getSubjectLink = function(url, subjectId) {
	if (subjectId) {
		var value = this.controller.getSubjectInfo(subjectId);
		return <a href={url + "/subject/" + subjectId}>{value}</a>;
	}

	return "";
}



ServiceRequestOverview.prototype.renderServiceRequests = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div class="card"/>;

	// Create page navigation

	var pagination = this.getPaginationDiv();

	var count = this.controller.getNumberOfServiceRequests();

	var homeURL = this.ui.homeURL(req.url);
	var url = homeURL + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}

	var pagenavigator = this.paginator.getPagination(count, url + "?op=changeoffset&");

	// Create table header

	var processFilter = this.getProcessFilter();
	var detailsFilter = this.getDetailsFilter();
	var lifeCycleFilter = this.getLifeCycleFilter();

	var table =
		<table class="pure-table pure-table-striped" style="table-layout:fixed">
			<thead>
				<tr>
					<th>ID</th>
					<th><label for="processfilter">Process</label></th>
					<th>Originator</th>
					<th>Recipient</th>
					<th><label for="detailsFilter">Details</label></th>
					<th><label for="lifeCycleFilter">Life-Cycle</label></th>
				</tr>
				<tr>
					<th></th>
					<th>{processFilter}</th>
					<th></th>
					<th></th>
					<th>{detailsFilter}</th>
					<th>{lifeCycleFilter}</th>
				</tr>
			</thead>
			<tbody/>
		</table>;

	var url = this.ui.homeURL(req.url);

	// Fill table

	var srlist = this.controller.getServiceRequests(count);

	for (var i = 0; i < srlist.length; i++) {
		var sr = srlist[i];

		var originator = this.getSubjectLink(url, sr.originatorId);
		var recipient = this.getSubjectLink(url, sr.recipientId);

		var details = sr.details
		if (!details) {
			var details = ""
		}

		var lifeCycle = ServiceRequestModel.getLifeCycleStringForId(sr.lifecycle);

		var tr = <tr>
				<td><a href={url + "/sr/" + sr.id}>{sr.id}</a></td>
				<td>{sr.process}</td>
				<td>{originator}</td>
				<td>{recipient}</td>
				<td>{details}</td>
				<td>{lifeCycle}</td>
			</tr>
 		table.tbody.appendChild(tr);
	}

	var tableDiv = <div class="pure-u-1">{table}</div>;
	pagination.appendChild(tableDiv);
	pagination.appendChild(pagenavigator);
	div.appendChild(pagination);
	return div;
}



ServiceRequestOverview.prototype.getOverview = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	if (this.controller.isAuditView) {
		var title = "Audit Service Requests";
	} else {
		var title = "Service Requests";
	}

	var page =	<div>
				<h1>{title}</h1>

				<br/>
				<div id="list" />
				<br/>
			</div>;

	var div = page..div.(@id == "list");

	var overview = this.renderServiceRequests(req, res);
	div.appendChild(overview);

	return page;
}
