/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2016 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Core service requests factory
 */


var BaseServiceRequest			= require('scsh/srv-cc1/BaseServiceRequest').BaseServiceRequest;

var CertificateRequestController	= require('pki-as-a-service/processes/CertificateRequestController').CertificateRequestController;
var CertificateRequestModel		= require('pki-as-a-service/processes/CertificateRequestModel').CertificateRequestModel;

var PKCS10RequestController		= require('pki-as-a-service/processes/PKCS10RequestController').PKCS10RequestController;
var PKCS10RequestModel			= require('pki-as-a-service/processes/PKCS10RequestModel').PKCS10RequestModel;

var ESTRequestController		= require('pki-as-a-service/processes/ESTRequestController').ESTRequestController;
var ESTRequestModel			= require('pki-as-a-service/processes/ESTRequestModel').ESTRequestModel;

var CreateESTServerRequestController	= require('pki-as-a-service/processes/CreateESTServerRequestController').CreateESTServerRequestController;
var CreateESTServerRequestModel		= require('pki-as-a-service/processes/CreateESTServerRequestModel').CreateESTServerRequestModel;

var ImportTrustedCertificatesRequestController = require('pki-as-a-service/processes/ImportTrustedCertificatesRequestController').ImportTrustedCertificatesRequestController;
var ImportTrustedCertificatesRequestModel = require('pki-as-a-service/processes/ImportTrustedCertificatesRequestModel').ImportTrustedCertificatesRequestModel;

var CreateTrustCenterRequestController	= require('pki-as-a-service/processes/CreateTrustCenterRequestController').CreateTrustCenterRequestController;
var CreateTrustCenterRequestModel	= require('pki-as-a-service/processes/CreateTrustCenterRequestModel').CreateTrustCenterRequestModel;

var JoinESTServerServiceRequestController = require('pki-as-a-service/processes/JoinESTServerServiceRequestController').JoinESTServerServiceRequestController;
var JoinESTServerServiceRequestModel	= require('pki-as-a-service/processes/JoinESTServerServiceRequestModel').JoinESTServerServiceRequestModel;

var CreateX509SignerRequestController	= require('pki-as-a-service/processes/CreateX509SignerRequestController').CreateX509SignerRequestController;
var CreateX509SignerRequestModel	= require('pki-as-a-service/processes/CreateX509SignerRequestModel').CreateX509SignerRequestModel;

var ImportX509SignerRequestController	= require('pki-as-a-service/processes/ImportX509SignerRequestController').ImportX509SignerRequestController;
var ImportX509SignerRequestModel	= require('pki-as-a-service/processes/ImportX509SignerRequestModel').ImportX509SignerRequestModel;

var IssueCRLRequestController		= require('pki-as-a-service/processes/IssueCRLRequestController').IssueCRLRequestController;
var IssueCRLRequestModel		= require('pki-as-a-service/processes/IssueCRLRequestModel').IssueCRLRequestModel;

var IssueTokenRequestModel		= require('pki-as-a-service/processes/IssueTokenRequestModel').IssueTokenRequestModel;
var IssueTokenRequestController		= require('pki-as-a-service/processes/IssueTokenRequestController').IssueTokenRequestController;

var ResetRetryCounterServiceRequestModel = require('pki-as-a-service/processes/ResetRetryCounterServiceRequestModel').ResetRetryCounterServiceRequestModel;
var ResetRetryCounterServiceRequestController = require('pki-as-a-service/processes/ResetRetryCounterServiceRequestController').ResetRetryCounterServiceRequestController;

var EnrollServiceRequestModel		= require('pki-as-a-service/processes/EnrollServiceRequestModel').EnrollServiceRequestModel;
var EnrollPersonRequestController	= require('pki-as-a-service/processes/EnrollPersonRequestController').EnrollPersonRequestController;
var EnrollSystemRequestController	= require('pki-as-a-service/processes/EnrollSystemRequestController').EnrollSystemRequestController;
var RegisterMyTokenRequestModel		= require('pki-as-a-service/processes/RegisterMyTokenRequestModel').RegisterMyTokenRequestModel;
var RegisterMyTokenRequestController	= require('pki-as-a-service/processes/RegisterMyTokenRequestController').RegisterMyTokenRequestController;
var ManageRoleServiceRequestModel	= require('pki-as-a-service/processes/ManageRoleServiceRequestModel').ManageRoleServiceRequestModel;
var ManageRoleServiceRequestController	= require('pki-as-a-service/processes/ManageRoleServiceRequestController').ManageRoleServiceRequestController;

var TrustCenter				= require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var EnrolledSubject			= require('pki-as-a-service/subjects/EnrolledSubject').EnrolledSubject;
var Person					= require('pki-as-a-service/subjects/Person').Person;

var RoleDAO				= require('scsh/pki-db/db/RoleDAO').RoleDAO;


function CoreServiceRequests(service) {
	this.service = service;
	this.priority = 500;
}

exports.Plugin = CoreServiceRequests;



CoreServiceRequests.prototype.getFactoryForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForProcess(" + process + ")");

	switch(process) {
		case "CertificateServiceRequest":
		case "PKCS10ServiceRequest":
		case "ESTServiceRequest":
		case "CreateESTServerServiceRequest":
		case "JoinESTServerServiceRequest":
		case "CreateTrustCenterServiceRequest":
		case "CreateX509SignerServiceRequest":
		case "ImportX509SignerServiceRequest":
		case "IssueCRLServiceRequest":
		case "IssueTokenServiceRequest":
		case "ImportTrustedCertificatesServiceRequest":
		case "EnrollPersonServiceRequest":
		case "EnrollSystemServiceRequest":
		case "RegisterMyTokenServiceRequest":
		case "ResetRetryCounterServiceRequest":
		case "ManageRoleServiceRequest":
			return this;
			break;
	}
	return undefined;
}



CoreServiceRequests.prototype.enumerateRunnableProcesses = function(user, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcesses()");

	processes.push( { label: "Request Certificate", process: "CertificateServiceRequest" } );
	processes.push( { label: "Request Certificate (PKCS#10)", process: "PKCS10ServiceRequest" } );

	if (user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
		processes.push( { label: "Create Trust Center", process: "CreateTrustCenterServiceRequest" } );
	}
}



CoreServiceRequests.prototype.enumerateRunnableProcessesForSubject = function(user, subject, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcessesForSubject()");

	if ((subject instanceof EnrolledSubject) && (subject.isManager(user))) {
		processes.push( { label: "Issue Token", process: "IssueTokenServiceRequest" } );
	}

	// If no role is assigned to the current user
	// then ManageRoleServiceRequest cannot
	// add or remove any role
	if (user.roles && user.roles.length > 0) {
		if ((subject instanceof Person) && (subject.canRead(user))) {
			processes.push( { label: "Manage Role", process: "ManageRoleServiceRequest" } );
		}
	}

	if (subject instanceof TrustCenter) {
		if (subject.isCertificationOfficer(user)) {
			processes.push( { label: "Create Signer", process: "CreateX509SignerServiceRequest" } );
			processes.push( { label: "Import Signer", process: "ImportX509SignerServiceRequest" } );
			processes.push( { label: "Issue CRL", process: "IssueCRLServiceRequest" } );
		}

		if (subject.isRegistrationOfficer(user)) {
			processes.push( { label: "Enroll System", process: "EnrollSystemServiceRequest" } );
			processes.push( { label: "Enroll Person", process: "EnrollPersonServiceRequest" } );
			processes.push( { label: "Import Trusted Certificates", process: "ImportTrustedCertificatesServiceRequest" } );
			processes.push( { label: "Join EST Server", process: "JoinESTServerServiceRequest" } );
		}

		if (subject.isManager(user)) {
			if (user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
				processes.push( { label: "Create EST Server", process: "CreateESTServerServiceRequest" } );
			}
		}
	}
}



CoreServiceRequests.prototype.getServiceRequest = function(bo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getServiceRequest(" + bo + ")");
	switch (bo.process) {
		case "EnrollPersonServiceRequest":
		case "EnrollSystemServiceRequest":
			var sr = new EnrollServiceRequestModel(this.service, bo);
			break;
		case "IssueTokenServiceRequest":
			var sr = new IssueTokenRequestModel(this.service, bo);
			break;
		case "ResetRetryCounterServiceRequest":
			var sr = new ResetRetryCounterServiceRequestModel(this.service, bo);
			break;
		case "RegisterMyTokenServiceRequest":
			var sr = new RegisterMyTokenRequestModel(this.service, bo);
			break;
		case "CertificateServiceRequest":
			var sr = new CertificateRequestModel(this.service, bo);
			break;
		case "PKCS10ServiceRequest":
			var sr = new PKCS10RequestModel(this.service, bo);
			break;
		case "ESTServiceRequest":
			var sr = new ESTRequestModel(this.service, bo);
			break;
		case "CreateESTServerServiceRequest":
			var sr = new CreateESTServerRequestModel(this.service, bo);
			break;
		case "JoinESTServerServiceRequest":
			var sr = new JoinESTServerServiceRequestModel(this.service, bo);
			break;
		case "ImportTrustedCertificatesServiceRequest":
			var sr = new ImportTrustedCertificatesRequestModel(this.service, bo);
			break;
		case "CreateTrustCenterServiceRequest":
			var sr = new CreateTrustCenterRequestModel(this.service, bo);
			break;
		case "CreateX509SignerServiceRequest":
			var sr = new CreateX509SignerRequestModel(this.service, bo);
			break;
		case "ImportX509SignerServiceRequest":
			var sr = new ImportX509SignerRequestModel(this.service, bo);
			break;
		case "IssueCRLServiceRequest":
			var sr = new IssueCRLRequestModel(this.service, bo);
			break;
		case "ManageRoleServiceRequest":
			var sr = new ManageRoleServiceRequestModel(this.service, bo);
			break;
		default:
			var sr = BaseServiceRequest.deserialize(bo.content);
			break;
	}
	sr.bo = bo;
	sr.id = sr.bo.id;
	sr.service = this.service;
	return sr;
}



CoreServiceRequests.prototype.getControllerForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForProcess(" + process + ")");
	switch(process) {
		case "CertificateServiceRequest":
			return new CertificateRequestController();
		case "PKCS10ServiceRequest":
			return new PKCS10RequestController();
		case "ESTServiceRequest":
			return new ESTRequestController();
		case "CreateESTServerServiceRequest":
			return new CreateESTServerRequestController();
		case "JoinESTServerServiceRequest":
			return new JoinESTServerServiceRequestController();
		case "ImportTrustedCertificatesServiceRequest":
			return new ImportTrustedCertificatesRequestController();
		case "CreateTrustCenterServiceRequest":
			return new CreateTrustCenterRequestController();
		case "CreateX509SignerServiceRequest":
			return new CreateX509SignerRequestController();
		case "ImportX509SignerServiceRequest":
			return new ImportX509SignerRequestController();
		case "IssueCRLServiceRequest":
			return new IssueCRLRequestController();
		case "IssueTokenServiceRequest":
			return new IssueTokenRequestController();
		case "ResetRetryCounterServiceRequest":
			return new ResetRetryCounterServiceRequestController();
		case "EnrollPersonServiceRequest":
			return new EnrollPersonRequestController();
		case "EnrollSystemServiceRequest":
			return new EnrollSystemRequestController();
		case "RegisterMyTokenServiceRequest":
			return new RegisterMyTokenRequestController();
		case "ManageRoleServiceRequest":
			return new ManageRoleServiceRequestController();
	}
	return undefined;
}



CoreServiceRequests.prototype.toString = function() {
	return "Core Service Requests";
}
