/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2022 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for JoinESTServerServiceRequest
 */

var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.joinest.title": "Join EST Server",
		"msg.joinest.estConfiguration": "EST Server Configuration",
		"msg.joinest.estServer": "EST Server",
		"msg.joinest.trustedCA": "Trusted CA",
		"msg.joinest.issuer": "Issuer",
		"msg.joinest.autoEnroll": "Enable Auto Enrollment",
		"msg.joinest.retryAfter": "Retry After (Seconds)",

		"action.joinest.deactivate": "Deactivate Association",
		"action.joinest.reactivate": "Reactivate Association",
	}}
]);



function JoinESTServerServiceRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new JoinESTServerServiceRequestController()");
}

exports.JoinESTServerServiceRequestController = JoinESTServerServiceRequestController;



JoinESTServerServiceRequestController.prototype.getSubject = function(req, res, srct) {
	var parameter = CommonUI.parseQueryString(req.queryString);
	var ca = Number(parameter.ca);

	if (isNaN(ca) || ca == 0) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Argument ca not valid in URL");
	}

	var subject = srct.service.getSubject(ca);

	if (!(subject instanceof TrustCenter)) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Subject is not a TrustCenter instance");
	}

	return subject;
}



JoinESTServerServiceRequestController.prototype.createRequest = function(srct, trustCenterId) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "JoinESTServerServiceRequest",
		title: srct.i18n("msg.joinest.title"),
		originatorId: trustCenterId,
		state: "Created",
		content: {
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



JoinESTServerServiceRequestController.prototype.postCreateAction = function(req, res, srct, action, trustCenter) {
	var bo = this.createRequest(srct, trustCenter.getId());

	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



JoinESTServerServiceRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.submit":
	case "action.save":
	case "action.estsubject.changeMode":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



JoinESTServerServiceRequestController.prototype.getCreateView = function(req, res, srct, trustCenter) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var form =	<div>
				<h1>Join EST-Server</h1>

				<p>An EST-Server enables the Enrollment over Secure Transport (EST) for a participating TrustCenter.</p>

				<p>This service request allows a Trustcenter to join an EST-Server in the system. By joining an
				   EST-Server, you can defined properties (i.e. trusted certificates, signer that can reenroll without RA approval)
				   for requests via the EST-Server and processes in the Trustcenter.</p>

				<p>An EST-Server can be either private or public. A public EST-Server can be joined
				   by any TrustCenter in the PKI-as-a-Service Portal.
				   A private EST-Server can only be joined by the managing TrustCenter.</p>

				<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
				</form>
			</div>

	return form;
}



JoinESTServerServiceRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	return div;
}
