/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model for requesting a certificate for a key on a SmartCard-HSM
 */

ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
CVC				= require('scsh/eac/CVC').CVC;
SmartCardHSM			= require('scsh/sc-hsm/SmartCardHSM').SmartCardHSM;
SmartCardHSMKeySpecGenerator	= require('scsh/sc-hsm/SmartCardHSM').SmartCardHSMKeySpecGenerator;
HSMKeyStore			= require("scsh/sc-hsm/HSMKeyStore").HSMKeyStore;
CryptoProvider			= require('scsh/sc-hsm/CryptoProvider').CryptoProvider;
DNEncoder			= require("scsh/x509/DNEncoder").DNEncoder;
PKIXCommon			= require("scsh/x509/PKIXCommon").PKIXCommon;
Holder				= require('scsh/pki-db/Holder').Holder;
X509Signer			= require('scsh/x509/X509Signer').X509Signer;
CRLGenerator			= require('scsh/x509/CRLGenerator').CRLGenerator;
File				= require('scsh/file/File').File;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function IssueCRLRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
}

IssueCRLRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
IssueCRLRequestModel.constructor = IssueCRLRequestModel;

exports.IssueCRLRequestModel = IssueCRLRequestModel;



IssueCRLRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		this.form = [];

		var holderDAO = this.service.daof.getHolderDAO();
		var caList = holderDAO.getHolderListBySubject(this.getRecipientId());
		var selectionModel = [];
		for (var i = 0; i < caList.length; i++) {
			var ca = caList[i];
			selectionModel.push( { id: ca.id, value: ca.name, selected: ca.id == this.model.holder } ) ;
		}

		var selectableHolder = this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT;
		var fieldset = {
			id: "crl",
			legend: "msg.crl.data",
			fields: [
				{
					id: "holder",
					label: "msg.crl.holder",
					type: "select",
					editable: selectableHolder,
					required: true,
					value: selectionModel
				}
			]
		};

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_COMPLETE) {
			fieldset.fields.push(
				{
					id: "crlNumber",
					label: "msg.crl.crlNumber",
					type: "number",
					editable: false,
					required: false,
					value: this.model.crlNumber
				});
			fieldset.fields.push(
				{
					id: "crlEntries",
					label: "msg.crl.crlEntries",
					type: "number",
					editable: false,
					required: false,
					value: this.model.crlEntries
				});
		}

		this.form.push(fieldset);
	}

	return this.form;
}



IssueCRLRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	if (!this.isCertificationOfficer(user)) {
		return this.actionList;
	}

	var ca = this.getRecipient();
	if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT) {
		this.actionList.push("action.crl.select");
	}
	if (this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_COMPLETE &&
		this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_EDIT) {
		if (this.isCATokenInHSMService()) {
			this.actionList.push("action.crl.create.hsmservice");
		}
	} else {
		if (ca.getCRLDistributionPoint()) {
			this.actionList.push("action.crl.download");
		}
	}

	return this.actionList;
}



IssueCRLRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.crl.select":
			this.setStatusInfo("CA selected");
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
			break;
		case "action.crl.create.hsmservice":
			this.issueCRL(this.service.hsmService, user);
			break;
		case "action.crl.create.usertoken":
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_PRODUCE);
			this.setStatusInfo("Generate Certificate Revocaction");
			break;
	}

	this.commit(user.id);
	return true;
}



IssueCRLRequestModel.prototype.isCATokenInHSMService = function() {
	var trustCenter = this.getRecipient();

	var ca = trustCenter.getSigner(this.model.holder);

	var cp = trustCenter.getCryptoProvider(ca.keyDomain);
	if (cp) {
		return true;
	}

	return false;
}



IssueCRLRequestModel.prototype.isCertificationOfficer = function(user) {
	return this.getRecipient().isCertificationOfficer(user);
}



IssueCRLRequestModel.prototype.issueCRL = function(cpf, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "issueCRL()");

	var trustCenter = this.getRecipient();

	// Create CRL
	var ca = trustCenter.getX509CertificateIssuer(this.model.holder);
	var crl = ca.issueCRL();

	// Process CRL
	var crlBin = crl.getBytes();
	var crlGen = new CRLGenerator();
	this.model.crlNumber = crlGen.loadCRLEntries(crlBin);
	this.model.crlEntries = crlGen.revokedCertificates.length;
	this.model.crl = crlBin.toString(HEX);

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
	this.setStatusInfo("Certificate Revocaction List published");

	this.commit(user.id);
}



IssueCRLRequestModel.prototype.handleSmartCardHSM = function(sc, chain, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleSmartCardHSM in lifecycle: " + this.bo.lifecycle);

	if (this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_COMPLETE) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_PRODUCE);
		this.setStatusInfo("Generate Certificate Revocaction");

		var provider = new CryptoProvider(sc, chain.path);
		var cpf = { getCryptoProvider: function() { return provider }};
		this.issueCRL(cpf, user);
	} else {
		GPSystem.log(GPSystem.ERROR, module.id, "Unexpected handleCard in state " + this.bo.lifecycle);
	}
}



IssueCRLRequestModel.prototype.handleCard = function(card, session, reqinfo) {
	var user = session.user;

	var sc = new SmartCardHSM(card);
	var devAutCert = sc.readBinary(SmartCardHSM.C_DevAut);
	var crypto = new Crypto();
	var chain = SmartCardHSM.validateCertificateChain(crypto, devAutCert);
	if (chain == null) {
		this.addMessage("Could not authenticate SmartCard-HSM");
		return;
	}

	sc.openSecureChannel(crypto, chain.publicKey);

	this.handleSmartCardHSM(sc, chain, user);
}
