/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for ImportX509SignerServiceRequest
 */

ServiceRequestModel = require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
ImportSignerWizard = require('pki-as-a-service/ui/ImportSignerWizard').ImportSignerWizard;
CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
SubjectDAO = require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;
CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.impsigner.title": "Import X509 CA Signer",

		"msg.impsigner.subjectspec": "Subject Certificate Definitions",
		"msg.impsigner.subKeySpecId": "Key Specification",
		"msg.impsigner.subCertValidityDays": "Expires after (days)",
		"msg.impsigner.dnmask": "Subject DN Rule",
		"msg.impsigner.supportedProfiles": "Supported Profiles",

		"action.impsigner.import": "Import Signer",

		"msg.impsigner.caPathLen": "Subordinate CAs",
		"msg.impsigner.crl": "CRL Definitions",
		"msg.impsigner.validityDaysCRL": "Expires after (days)"
// 		"action.xsigner.produce.hsmservice": "Issue Certificate",
// 		"action.xsigner.createcrl": "Create CRL"
	}}
//	,
//	{ lang: "DE", keys: {
//		"action.save": "Speichern"
//	}}
]);



function ImportX509SignerRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new ImportX509SignerRequestController()");
}

exports.ImportX509SignerRequestController = ImportX509SignerRequestController;



ImportX509SignerRequestController.prototype.getSubject = function(req, res, srct) {
	var parameter = CommonUI.parseQueryString(req.queryString);
	var ca = Number(parameter.ca);

	if (isNaN(ca) || ca == 0) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Argument ca not valid in URL");
	}

	var subject = srct.service.getSubject(ca);

	return subject;
}



ImportX509SignerRequestController.prototype.createRequest = function(srct, caSubjectId) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "ImportX509SignerServiceRequest",
		title: srct.i18n("msg.impsigner.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: caSubjectId,
		state: "Created",
		content: {
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



ImportX509SignerRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	var ca = this.getSubject(req, res, srct);

	if (!ca.isManager(srct.ui.session.user) &&
		!ca.isCertificationOfficer(srct.ui.session.user)) {
		return false;
	}

	var bo = this.createRequest(srct, ca.getId());

	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



ImportX509SignerRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {
	print("handleAction: " + action);

	switch(action) {
	case "action.submit":
	case "action.save":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	case "action.impsigner.import":
		srct.ui.wizard = new ImportSignerWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	case "action.impsigner.enumerate":
		srct.ui.wizard = new CardActionWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



ImportX509SignerRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var ca = this.getSubject(req, res, srct);
	var subjectLink = srct.ui.homeURL(req.url) + "/subject/" + ca.getId();

	var form =	<div>
				<h1>Import X509 Signer</h1>
				<p>Import an X509 signer from a given token and assign it to CA <a href={subjectLink}>{ca.getName()}</a>.</p>
			</div>

	if (ca.isManager(srct.ui.session.user) ||
		ca.isCertificationOfficer(srct.ui.session.user)) {
		form.appendChild(
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		);
	} else {
		form.appendChild(
			<p>You are currently not authorized to import a signer for <a href={subjectLink}>{"CA " + ca.getName()}</a>.</p>
		)
	}

	return form;
}



ImportX509SignerRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	if (sr.getLifeCycle() >= ServiceRequestModel.LIFECYCLE_INUSE) {
		var certView = srct.renderCertificateList(req, res, sr);
		if (certView) {
			div.appendChild( <br/> );
			div.appendChild( <h2>Certificates</h2> );
			div.appendChild( certView );
		}
	}

	return div;
}
