/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2016 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview SubjectModel
 */

RoleDAO				= require('scsh/pki-db/db/RoleDAO').RoleDAO;
SubjectDAO			= require('scsh/pki-db/db/SubjectDAO').SubjectDAO;

function SubjectModel(service, bo) {
	this.service = service;
	this.bo = bo;
}

exports.SubjectModel = SubjectModel;



SubjectModel.prototype.getId = function() {
	return this.bo.id;
}



SubjectModel.prototype.getType = function() {
	return this.bo.type;
}



SubjectModel.prototype.getName = function() {
	return this.bo.name;
}



SubjectModel.prototype.getEmail = function() {
	return this.bo.email;
}



SubjectModel.prototype.getLifeCycle = function() {
	return this.bo.lifecycle;
}



SubjectModel.prototype.setLifeCycle = function(lifeCycle) {
	GPSystem.log(GPSystem.DEBUG, module.id, "setLifeCycle(" + lifeCycle + ")");
	if (this.bo.lifecycle == SubjectDAO.LIFECYCLE_TERMINATED) {
		return false;
	}

	if (typeof(lifeCycle) == "string") {
		lifeCycle = parseInt(lifeCycle);
	}

	if (this.bo.lifecycle == lifeCycle) {
		return false;
	}

	var subjectDAO = this.service.daof.getSubjectDAO();
	subjectDAO.updateLifeCycle(this.bo, lifeCycle);

	return true;
}



/**
 * Return a map of keys that can be used in DN transformations
 *
 * @type Object
 * @return a map of key/value strings that can be used in DN transformations
 */
SubjectModel.prototype.getSubjectMap = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "getSubjectMap()");

	var map = {
		email: this.bo.email,
		name: this.bo.name
	}

	return map;
}



SubjectModel.prototype.getToken = function() {
	var tokenDAO = this.service.daof.getTokenDAO();
	var list = tokenDAO.getTokenListBySubjectId(this.getId());

	return list;
}



SubjectModel.prototype.isManager = function(user) {
	return (this.bo.managedByRoleId && (user.roles.indexOf(this.bo.managedByRoleId) >= 0));
}



SubjectModel.prototype.isOwner = function(user) {
	return (user.id == this.bo.id)
}



SubjectModel.prototype.canRead = function(user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "canRead()");
	if (this.isOwner(user)) {
		GPSystem.log(GPSystem.DEBUG, module.id, "canRead() : isOwner()");
		return true;
	}

	switch (this.getLifeCycle()) {
		case SubjectDAO.LIFECYCLE_PUBLIC:
			GPSystem.log(GPSystem.DEBUG, module.id, "canRead() : PUBLIC");
			return true;
		case SubjectDAO.LIFECYCLE_SUSPENDED:
		case SubjectDAO.LIFECYCLE_TERMINATED:
			return false;
	}

	if (typeof(user.roles) != "undefined") {
		// Auditor Access
		if (user.roles.indexOf(RoleDAO.ID_Auditor) >= 0) {
			GPSystem.log(GPSystem.DEBUG, module.id, "canRead() : isAuditor");
			return true;
		}

		// Manager Access
		if (this.isManager(user)) {
			GPSystem.log(GPSystem.DEBUG, module.id, "canRead() : isManager");
			return true;
		}

		// Same Group
		if (this.bo.managedByRoleId && this.bo.managedByRoleId == user.managedByRoleId) {
			GPSystem.log(GPSystem.DEBUG, module.id, "canRead() : same group");
			return true;
		}
	}

	return false;
}



/**
 * Called whenever a new service request addressing the subject is created
 *
 * This call allows the subject to veto the creation or to modify the
 * database that is going to be created.
 *
 * @param {User} user the user object
 * @param {Object} object the template containing database entries for the new service request
 * @type Boolean
 * @return true if the caller can proceed
 */
SubjectModel.prototype.checkServiceRequestCreation = function(user, template) {
	return true;
}
