/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A service request creating a new CA signer
 */

ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;

SubjectDAO			= require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
SmartCardHSM			= require('scsh/sc-hsm/SmartCardHSM').SmartCardHSM;
// KeyDomain			= require('pki-as-a-service/service/KeyDomain').KeyDomain;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function CreateCASignerRequestModel(service, bo, type) {
	ServiceRequestModel.call(this, service, bo);
	this.validationResult = [];
	this.subjectType = type;
}

CreateCASignerRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
CreateCASignerRequestModel.constructor = CreateCASignerRequestModel;

exports.CreateCASignerRequestModel = CreateCASignerRequestModel;



CreateCASignerRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		var editable = (user.id == this.getOriginatorId()) &&
			((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUBMIT));


		var subjectDAO = this.service.daof.getSubjectDAO();
		var ca = subjectDAO.getSubject(this.getRecipientId());

		var ca = this.getRecipient();
		var tokenList = ca.getToken();
		var selectionList = [];
		for (var i = 0; i < tokenList.length; i++) {
			var token = tokenList[i];
			var hsm = this.service.hsmService.getHSMState(token.path);
			if (!hsm || hsm.isOffline() || hsm.error) {
				GPSystem.log(GPSystem.DEBUG, module.id, "unable to get hsm for path " + token.path);
				continue;
			}

			var optgroup = {
				optgroup: true,
				label: token.path,
				value: []
			};

			for (var j = 0; j < hsm.keyDomain.length; j++) {
				var kdid = hsm.keyDomain[j].toString(HEX);

				var isSelected = kdid.equals(this.model.keyDomain);

				if (kdid.length > 16) {
					kdid = kdid.substring(0, 16) + "...";
				}

				optgroup.value.push( { id: j, value: kdid, selected: isSelected } ) ;
			}
			selectionList.push( optgroup );
		}

		this.form = [{
			id: "subjectData",
			legend: "msg.signer.cadata",
			fields: [
				{ id: "caName", label: "msg.signer.caName", type: "text", size: 50, required: false, editable: false, value: ca.getName() }
			]}
		];

		var keyDomain = {
			id: "keySpec",
			legend: "msg.signer.keySpec",
			fields: []
		};

		if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT &&
			selectionList.length > 0) {
			keyDomain.fields.push({ id: "keyDomain", label: "msg.signer.selectDomain", type: "select", editable: editable, value: selectionList });
		} else if (this.model.keyDomain && isNaN(this.model.keyDomain)) {
			var kdid = this.model.keyDomain;
			if (kdid.length > 16) {
				kdid = kdid.substring(0, 16) + "...";
			}

			keyDomain.fields.push(
				{ id: "keyDomain", label: "msg.signer.selectDomain", type: "text", size: 50, required: false, editable: false, value: kdid }
			);
		}

		if (keyDomain.fields.length > 0) {
			this.form.push(keyDomain);
		}

	}

	return this.form;
}



CreateCASignerRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	if (user.id == this.getOriginatorId()) {
		if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT)) {
			this.actionList.push("action.save");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			if (this.model.keyDomain || typeof(this.model.keyDomain) == "number") {
				this.actionList.push("action.submit");
			}
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUBMIT) {
			if (this.findTokenWithDomain()) {
				this.actionList.push("action.signer.create");
			}
		}
	}

	return this.actionList;
}



CreateCASignerRequestModel.prototype.findTokenWithDomain = function(ca, id) {
	GPSystem.log(GPSystem.DEBUG, module.id, "findTokenWithDomain(" + ca + ", " + id + ")");
	if (!ca) {
		var ca = this.getRecipient();
	}

	var tokenList = ca.getToken();
	var hsm = null;
	var kdSlot = null;

	if (!id) {
		var id = new ByteString(this.model.keyDomain, HEX);
	}

	for (var i = 0; i < tokenList.length; i++) {
		var token = tokenList[i];
		hsm = this.service.hsmService.getHSMState(token.path);
		if (!hsm || hsm.isOffline()) {
			continue;
		}

		for (var j = 0; j < hsm.keyDomain.length; j++) {
			var kd = hsm.keyDomain[j];
			if (kd.equals(id)) {
				return {
					sc: hsm.sc,
					slot: j - 1 // -1 if default domain, otherwise the corresponding slot
				}
			}
		}
	}

	return null;
}



CreateCASignerRequestModel.prototype.validateField = function(fld, val) {
	switch(fld.id) {
		default:
			return this.validateFieldDefault(fld, val);
	}
	return true;
}



CreateCASignerRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		GPSystem.log(GPSystem.DEBUG, module.id, "action list :" + actionList);
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.save":
			this.save()
			break;
		case "action.submit":
			this.save()
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_SUBMIT);
			this.setStatusInfo("Request ready for signer creation");
			break;
		case "action.signer.create":
			this.createSigner(user);
			break;

	}

	this.commit(user.id);
	return true;
}



CreateCASignerRequestModel.prototype.save = function() {
	var commitChanges = false;

	if (this.getLifeCycle() == ServiceRequestModel.LIFECYCLE_NEW) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
		this.setStatusInfo("Form saved");
		commitChanges = true;
	}

	if (!isNaN(this.model.keyDomain)) {
		var ca = this.getRecipient();
		var tokenList = ca.getToken();
		var numberOfKD = 0;
		for (var i = 0; i < tokenList.length; i++) {
			var token = tokenList[i];
			var hsm = this.service.hsmService.getHSMState(token.path);
			if (!hsm || hsm.isOffline() || hsm.error) {
				GPSystem.log(GPSystem.DEBUG, module.id, "unable to get hsm for path " + token.path);
				continue;
			}

			// this.model.keyDomain is the zero based index of the selected key domain over all connected token.
			var offset = this.model.keyDomain - numberOfKD;

			if (hsm.keyDomain.length <= offset) {
				numberOfKD += hsm.keyDomain.length;
				GPSystem.log(GPSystem.DEBUG, module.id, "skip hsm " + token.path);
				continue
			}

			var kdid = hsm.keyDomain[offset].toString(HEX);
			this.model.keyDomain = kdid;

			commitChanges = true;
			break;
		}
	}

	commitChanges |= this.updateDetails(this.model.name);

	return commitChanges;
}
