/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2016 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Core service requests
 */


BaseServiceRequest			= require('scsh/srv-cc1/BaseServiceRequest').BaseServiceRequest;

CertificateRequestController		= require('pki-as-a-service/processes/CertificateRequestController').CertificateRequestController;
CertificateRequestModel			= require('pki-as-a-service/processes/CertificateRequestModel').CertificateRequestModel;

PKCS10RequestController		= require('pki-as-a-service/processes/PKCS10RequestController').PKCS10RequestController;
PKCS10RequestModel		= require('pki-as-a-service/processes/PKCS10RequestModel').PKCS10RequestModel;

CreateTrustCenterRequestController	= require('pki-as-a-service/processes/CreateTrustCenterRequestController').CreateTrustCenterRequestController;
CreateTrustCenterRequestModel		= require('pki-as-a-service/processes/CreateTrustCenterRequestModel').CreateTrustCenterRequestModel;

CreateX509SignerRequestController	= require('pki-as-a-service/processes/CreateX509SignerRequestController').CreateX509SignerRequestController;
CreateX509SignerRequestModel		= require('pki-as-a-service/processes/CreateX509SignerRequestModel').CreateX509SignerRequestModel;

ImportX509SignerRequestController	= require('pki-as-a-service/processes/ImportX509SignerRequestController').ImportX509SignerRequestController;
ImportX509SignerRequestModel		= require('pki-as-a-service/processes/ImportX509SignerRequestModel').ImportX509SignerRequestModel;

IssueCRLRequestController		= require('pki-as-a-service/processes/IssueCRLRequestController').IssueCRLRequestController;
IssueCRLRequestModel			= require('pki-as-a-service/processes/IssueCRLRequestModel').IssueCRLRequestModel;

IssueTokenRequestModel			= require('pki-as-a-service/processes/IssueTokenRequestModel').IssueTokenRequestModel;
IssueTokenRequestController		= require('pki-as-a-service/processes/IssueTokenRequestController').IssueTokenRequestController;

EnrollServiceRequestModel		= require('pki-as-a-service/processes/EnrollServiceRequestModel').EnrollServiceRequestModel;
EnrollPersonRequestController		= require('pki-as-a-service/processes/EnrollPersonRequestController').EnrollPersonRequestController;
EnrollSystemRequestController		= require('pki-as-a-service/processes/EnrollSystemRequestController').EnrollSystemRequestController;
RegisterMyTokenRequestModel		= require('pki-as-a-service/processes/RegisterMyTokenRequestModel').RegisterMyTokenRequestModel;
RegisterMyTokenRequestController	= require('pki-as-a-service/processes/RegisterMyTokenRequestController').RegisterMyTokenRequestController;
AssignRoleServiceRequestModel		= require('pki-as-a-service/processes/AssignRoleServiceRequestModel').AssignRoleServiceRequestModel;
AssignRoleServiceRequestController	= require('pki-as-a-service/processes/AssignRoleServiceRequestController').AssignRoleServiceRequestController;
ManageRoleServiceRequestModel		= require('pki-as-a-service/processes/ManageRoleServiceRequestModel').ManageRoleServiceRequestModel;
ManageRoleServiceRequestController	= require('pki-as-a-service/processes/ManageRoleServiceRequestController').ManageRoleServiceRequestController;


RoleDAO				= require('scsh/pki-db/db/RoleDAO').RoleDAO;


function CoreServiceRequests(service) {
	this.service = service;
	this.priority = 500;
}

exports.Plugin = CoreServiceRequests;



CoreServiceRequests.prototype.getFactoryForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForProcess(" + process + ")");

	switch(process) {
		case "CertificateServiceRequest":
		case "PKCS10ServiceRequest":
		case "CreateTrustCenterServiceRequest":
		case "CreateX509SignerServiceRequest":
		case "ImportX509SignerServiceRequest":
		case "IssueCRLServiceRequest":
		case "IssueTokenServiceRequest":
		case "EnrollPersonServiceRequest":
		case "EnrollSystemServiceRequest":
		case "RegisterMyTokenServiceRequest":
		case "AssignRoleServiceRequest":
		case "ManageRoleServiceRequest":
			return this;
			break;
	}
	return undefined;
}



CoreServiceRequests.prototype.enumerateRunnableProcesses = function(user, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcesses()");

	processes.push( { label: "Request Certificate", process: "CertificateServiceRequest" } );
	processes.push( { label: "Request Certificate (PKCS#10)", process: "PKCS10ServiceRequest" } );

	if (user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
		processes.push( { label: "Create Trust Center Subject", process: "CreateTrustCenterServiceRequest" } );
	}
}



CoreServiceRequests.prototype.getServiceRequest = function(bo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getServiceRequest(" + bo + ")");
	switch (bo.process) {
		case "EnrollPersonServiceRequest":
		case "EnrollSystemServiceRequest":
			var sr = new EnrollServiceRequestModel(this.service, bo);
			break;
		case "IssueTokenServiceRequest":
			var sr = new IssueTokenRequestModel(this.service, bo);
			break;
		case "RegisterMyTokenServiceRequest":
			var sr = new RegisterMyTokenRequestModel(this.service, bo);
			break;
		case "CertificateServiceRequest":
			var sr = new CertificateRequestModel(this.service, bo);
			break;
		case "PKCS10ServiceRequest":
			var sr = new PKCS10RequestModel(this.service, bo);
			break;
		case "CreateTrustCenterServiceRequest":
			var sr = new CreateTrustCenterRequestModel(this.service, bo);
			break;
		case "CreateX509SignerServiceRequest":
			var sr = new CreateX509SignerRequestModel(this.service, bo);
			break;
		case "ImportX509SignerServiceRequest":
			var sr = new ImportX509SignerRequestModel(this.service, bo);
			break;
		case "IssueCRLServiceRequest":
			var sr = new IssueCRLRequestModel(this.service, bo);
			break;
		case "AssignRoleServiceRequest":
			var sr = new AssignRoleServiceRequestModel(this.service, bo);
			break;
		case "ManageRoleServiceRequest":
			var sr = new ManageRoleServiceRequestModel(this.service, bo);
			break;
		default:
			var sr = BaseServiceRequest.deserialize(bo.content);
			break;
	}
	sr.bo = bo;
	sr.id = sr.bo.id;
	sr.service = this.service;
	return sr;
}



CoreServiceRequests.prototype.getControllerForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForProcess(" + process + ")");
	switch(process) {
		case "CertificateServiceRequest":
			return new CertificateRequestController();
		case "PKCS10ServiceRequest":
			return new PKCS10RequestController();
		case "CreateTrustCenterServiceRequest":
			return new CreateTrustCenterRequestController();
		case "CreateX509SignerServiceRequest":
			return new CreateX509SignerRequestController();
		case "ImportX509SignerServiceRequest":
			return new ImportX509SignerRequestController();
		case "IssueCRLServiceRequest":
			return new IssueCRLRequestController();
			break;
		case "IssueTokenServiceRequest":
			return new IssueTokenRequestController();
		case "EnrollPersonServiceRequest":
			return new EnrollPersonRequestController();
		case "EnrollSystemServiceRequest":
			return new EnrollSystemRequestController();
		case "RegisterMyTokenServiceRequest":
			return new RegisterMyTokenRequestController();
		case "AssignRoleServiceRequest":
			return new AssignRoleServiceRequestController();
		case "ManageRoleServiceRequest":
			return new ManageRoleServiceRequestController();
	}
	return undefined;
}



CoreServiceRequests.prototype.toString = function() {
	return "Core Service Requests";
}
