/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2017 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model for requesting a certificate for a key on a SmartCard-HSM
 */

ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function AssignRoleServiceRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
	this.validationResult = [];
}

AssignRoleServiceRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
AssignRoleServiceRequestModel.constructor = AssignRoleServiceRequestModel;

exports.AssignRoleServiceRequestModel = AssignRoleServiceRequestModel;



AssignRoleServiceRequestModel.prototype.getAssignableRoles = function() {
	var roleDAO = this.service.daof.getRoleDAO();

	var roleList = [];
	var assignableRoles = roleDAO.getManagedRoles(this.getOriginatorId());
	var recipientRoles = roleDAO.getAssignedRoleIds(this.getRecipientId());

	for (var i = 0; i < assignableRoles.length; i++) {
		var assignableRole = assignableRoles[i];
		if (recipientRoles.indexOf(assignableRole.id) == -1) {
			roleList.push(assignableRole);
		}
	}

	return roleList;
}



AssignRoleServiceRequestModel.prototype.assignSelectedRoles = function() {
	var roleDAO = this.service.daof.getRoleDAO();
	for (var i = 0; i < this.model.selectedRoles.length; i++) {
		var roleId = this.model.selectedRoles[i];
		roleDAO.newAssignedRole(this.getRecipientId(), roleId, this.bo.id);

		if(!this.model.assignedRoles) {
			this.model.assignedRoles = [];
		}
		this.model.assignedRoles.push(roleId);
	}
}



AssignRoleServiceRequestModel.prototype.updateDetails = function() {
	var roleDAO = this.service.daof.getRoleDAO();
	var assignedList = [];
	for (var i = 0; i < this.model.assignedRoles.length; i++) {
		var id = this.model.assignedRoles[i];
		var role = roleDAO.getRoleById(id);
		assignedList.push(role.name);
	}

	var details = "";
	if (assignedList.length > 0) {
		details = "" + assignedList;
	}

	if (details == this.bo.details) {
		return;
	}

	var srDAO = this.service.daof.getServiceRequestDAO();
	srDAO.updateDetails(this.bo, details);
}



AssignRoleServiceRequestModel.prototype.revokeSelectedRoles = function() {
	var roleDAO = this.service.daof.getRoleDAO();
	for (var i = 0; i < this.model.selectedRoles.length; i++) {
		var roleId = this.model.selectedRoles[i];
		roleDAO.deleteAssignedRole(this.getRecipientId(), roleId);

		var index = this.model.assignedRoles.indexOf(roleId);
		this.model.assignedRoles.splice(index, 1);
	}
}



AssignRoleServiceRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {

		if (!this.model.selectedRoles) {
			this.model.selectedRoles = []
		}
		if (!this.model.assignedRoles) {
			this.model.assignedRoles = [];
		}



		var subjectField = { id: "subject", label: "msg.ar.subject", type: "text", size: 50, required: true, editable: false, value: this.getRecipientId() };

		if (this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_COMPLETE) {
			var editable = true;

			var roles = this.getAssignableRoles();
			var roleList = [];
			for (var i = 0; i < roles.length; i++) {
				var role = roles[i];

				var e = { id: role.id, value: role.name };
				if (this.model.selectedRoles.indexOf(role.id) >= 0) {
					e.selected = true;
				}
				roleList.push(e);
			}

			if (roleList.length == 0) {
				roleList.push( {id: 1, msg: "msg.ar.noAssignableRoles"} );
				editable = false;
			}

			this.form = [{
				id: "applicant",
				legend: "msg.ar.ardata",
				fields: [
					subjectField,
					{ id: "selectedRoles", label: "msg.ar.roles", type: "select", multiselect: true, editable: editable, required: true, value: roleList }
				]}
			];
		} else {
			this.form = [{
				id: "applicant",
				legend: "msg.ar.assigned",
				fields: [ subjectField ]
			}];

			if (this.model.assignedRoles.length == 0) {
				var infoList = [ {id: 1, msg: "msg.ar.allRevoked"} ];
				this.form[0].fields.push(
					{ id: "selectedRoles", label: "msg.ar.roles", type: "select", multiselect: true, editable: false, value: infoList }
				);
			} else {
				var roleDAO = this.service.daof.getRoleDAO();
				var assignedList = [];
				for (var i = 0; i < this.model.assignedRoles.length; i++) {
					var id = this.model.assignedRoles[i];
					var role = roleDAO.getRoleById(id);
					assignedList.push( { id: role.id, value: role.name} );
				}
				this.form[0].fields.push(
					{ id: "selectedRoles", label: "msg.ar.roles", type: "select", multiselect: true, editable: true, required: true, value: assignedList }
				);
			}
		}

	}

	return this.form;
}



AssignRoleServiceRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	if (user.id == this.getOriginatorId()) {
		if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) || (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT)) {
			if (this.getAssignableRoles().length > 0) {
				this.actionList.push("action.save");
			}
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			this.actionList.push("action.assign");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_COMPLETE
			&& this.model.assignedRoles.length > 0) {
			this.actionList.push("action.revoke");
		}
	}

	return this.actionList;
}



AssignRoleServiceRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.save":
			if (this.getLifeCycle() == ServiceRequestModel.LIFECYCLE_NEW) {
				this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
				this.setStatusInfo("Request data entered");
			}
			break;
		case "action.assign":
			this.assignSelectedRoles();
			this.model.selectedRoles = [];
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
			this.setStatusInfo("Role has been assigned");
			this.updateDetails();
			break;
		case "action.revoke":
			this.revokeSelectedRoles();
			this.model.selectedRoles = [];
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
			this.setStatusInfo("Role has been revoked");
			this.updateDetails();
			break;
	}

	this.commit(user.id);
	return true;
}