/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A view for service requests
 */



function SubjectOverview(controller) {
	this.controller = controller;
	this.paginator = controller.paginator;
	this.ui = controller.ui;

	this.typeFilterValue = "";
	this.nameFilterValue = "";
	this.emailFilterValue = "";
}

exports.SubjectOverview = SubjectOverview;



SubjectOverview.SELECT_ANY_VALUE = "Any";



SubjectOverview.prototype.getTypeFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var typeFilter = <form class="pure-form" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<input name="op" type="hidden" value="filter"/>
		<select id="typeFilter" name="type" class="pure-u-23-24"/>
		<noscript><button class="pure-button" type="submit">Filter</button></noscript>
	</form>

	var selectType = typeFilter..select.(@id == "typeFilter");

	var option = <option>{SubjectOverview.SELECT_ANY_VALUE}</option>;
	if (this.typeFilterValue == "") {
		option.@selected = "selected";
	}
	selectType.appendChild(option);

	var types = this.controller.getSubjectTypes();
	for (var i = 0; i < types.length; i++) {
		var option = <option>{types[i]}</option>;
		if (this.typeFilterValue == types[i]) {
			option.@selected = "selected";
		}
		selectType.appendChild(option);
	}

	return typeFilter;
}



SubjectOverview.prototype.getNamesFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";


	var nameFilter = <form class="pure-form" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<input name="op" type="hidden" value="filter"/>
		<input id="nameFilter" type="search" placeholder="Search for Names" class="pure-u-23-24" name="name" value={this.nameFilterValue}/>
		<noscript><button class="pure-button" type="submit">Filter</button></noscript>
	</form>

	return nameFilter;
}



SubjectOverview.prototype.getEmailsFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";


	var emailFilter = <form class="pure-form" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<input name="op" type="hidden" value="filter"/>
		<input id="emailFilter" type="search" placeholder="Search for E-mails" class="pure-u-23-24" name="email" value={this.emailFilterValue}/>
		<noscript><button class="pure-button" type="submit">Filter</button></noscript>
	</form>

	return emailFilter;
}



SubjectOverview.prototype.getPaginationDiv = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var pagination =
		<div class="table-control">
			<div class="">
				<form class="pure-form" action="" method="get" onchange="submit()">
					<input name="op" type="hidden" value="changepagesize"/>
					<fieldset>
						<label for="size">Table Size:</label>
						<select id="size" name="pagesize"/>
						<noscript><button class="pure-button" type="submit">Change</button></noscript>
					</fieldset>
				</form>
			</div>
		</div>;

	var selectSize = pagination..select.(@id == "size");

	var option = <option>5</option>;
	if (this.paginator.pageSize == 5) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>10</option>;
	if (this.paginator.pageSize == 10) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>20</option>;
	if (this.paginator.pageSize == 20) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>50</option>;
	if (this.paginator.pageSize == 50) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>100</option>;
	if (this.paginator.pageSize == 100) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	return pagination;
}



SubjectOverview.prototype.renderSubjects = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div class="card"/>;

	// Create page navigation

	var pagination = this.getPaginationDiv();

	var count = this.controller.getNumberOfSubjects();

	var homeURL = this.ui.homeURL(req.url);
	var url = homeURL + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}
	var pagenavigator = this.paginator.getPagination(count, url + "?op=changeoffset&");

	// Create table header

	var typeFilter = this.getTypeFilter();
	var nameFilter = this.getNamesFilter();
	var emailFilter = this.getEmailsFilter();

	var table =
		<table class="pure-table pure-table-striped" style="table-layout:fixed">
			<thead>
				<tr>
					<th>ID</th>
					<th><label for="typeFilter">Type</label></th>
					<th><label for="nameFilter">Name</label></th>
					<th><label for="emailFilter">E-mail</label></th>
				</tr>
				<tr>
					<th></th>
					<th>{typeFilter}</th>
					<th>{nameFilter}</th>
					<th>{emailFilter}</th>
				</tr>
			</thead>
			<tbody/>
		</table>;

	// Fill table

	var list = this.controller.getSubjects(count);

	for (var i = 0; i < list.length; i++) {
		var subject = list[i];

		var id = <a href={homeURL + "/subject/" + subject.id}>{subject.id}</a>;
		var name = subject.name ? subject.name : "";
		var email = subject.email ? subject.email : "";

		var tr = <tr>
				<td>{id}</td>
				<td>{subject.type}</td>
				<td>{name}</td>
				<td>{email}</td>
			</tr>
 		table.tbody.appendChild(tr);
	}

	var tableDiv = <div class="pure-u-1">{table}</div>;
	pagination.appendChild(tableDiv);
	pagination.appendChild(pagenavigator);
	div.appendChild(pagination);
	return div;
}



SubjectOverview.prototype.getOverview = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	if (this.controller.isAuditView) {
		var title = "Audit Subjects";
	} else {
		var title = "Subjects";
	}

	var page =	<div>
				<h1>{title}</h1>

				<br/>
				<div id="list" />
				<br/>
			</div>;

	var div = page..div.(@id == "list");

	var overview = this.renderSubjects(req, res);
	div.appendChild(overview);

	return page;
}
