/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview An overview for subjects
 */

var CommonUI			= require('scsh/srv-cc1/CommonUI').CommonUI;
var Paginator			= require('pki-as-a-service/ui/Paginator').Paginator;
var SubjectOverview		= require('pki-as-a-service/ui/SubjectOverview').SubjectOverview;



function SubjectOverviewController(ui, isAuditView) {
	this.ui = ui;
	this.service = ui.service;
	this.isAuditView = isAuditView;

	this.paginator = new Paginator();

	this.filter = {};
	this.typeFilterValue = "";
	this.nameFilterValue = "";
	this.emailFilterValue = "";

	this.view = new SubjectOverview(this);
}

exports.SubjectOverviewController = SubjectOverviewController;



SubjectOverviewController.SELECT_ANY_VALUE = "Any";



SubjectOverviewController.prototype.handleRequest = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleRequest()");
	GPSystem.log(GPSystem.DEBUG, module.id, "req.method:" + req.method);
	GPSystem.log(GPSystem.DEBUG, module.id, "req.queryString:" + req.queryString);

	if (req.queryString) {
		this.handleOperation(req, res);
	}

	if (!this.isAuditView && req.url.length != 2) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Invalid URL");
	}

	if (req.parameters) {
		this.handlePostOperation(req, res);
	}

	var page = this.getPage(req, res);
	this.ui.sendPage(req, res, req.url, page);
}



SubjectOverviewController.prototype.handleOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleOperation(" + req.queryString + ")");
	var operation = CommonUI.parseQueryString(req.queryString);

	switch(operation.op) {
		case "changepagesize":
			var val = parseInt(operation.pagesize);
			if (val > 0) {
				this.paginator.pageSize = val;
			}
			if (val > this.paginator.offset) {
				this.paginator.offset = 0;
			}
			break;
		case "changeoffset":
			var val = parseInt(operation.offset);
			this.paginator.offset = val;
			break;
	}

	var url = this.ui.homeURL(req.url) + "/" + req.url[1];
	if (req.url.length > 2) {
		url += "/" + req.url[2];
	}

	res.setStatus(HttpResponse.SC_SEE_OTHER);
	res.addHeader("Location", url);
}



SubjectOverviewController.prototype.handlePostOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handlePostOperation( op=" + req.parameters.op + " )");
	if (req.parameters.op == "filter") {
		if (req.parameters.type) {
			this.updateTypeFilter(req.parameters.type);
		}

		if (typeof(req.parameters.name) != "undefined") {
			this.updateNameFilter(req.parameters.name);
		}

		if (typeof(req.parameters.email) != "undefined") {
			this.updateEmailFilter(req.parameters.email);
		}
	}
}



SubjectOverviewController.prototype.getPage = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = this.view.getOverview(req, res);

	var t = this.ui.generateTemplate(req.url);
 	var c = t..div.(@id == "content");
 	c.div = page;

	return t;
}



SubjectOverviewController.prototype.getSubjectTypes = function(req, res) {
	var dao = this.service.daof.getSubjectDAO();
	return dao.listSubjectTypes();
}



SubjectOverviewController.prototype.getNumberOfSubjects = function() {
	var dao = this.service.daof.getSubjectDAO();

	if (this.isAuditView) {
		var records = dao.countSubjectsByFilter(this.filter);
	} else {
		var records = dao.countManagedSubjects(
			this.ui.session.user.id,
			this.ui.session.user.managedByRoleId,
			this.view.typeFilterValue,
			this.view.nameFilterValue,
			this.view.emailFilterValue);
	}

	return records;
}



SubjectOverviewController.prototype.getSubjects = function(count) {
	var dao = this.service.daof.getSubjectDAO();

	if (this.isAuditView) {
		var subjects = dao.listSubjectsByFilter(
			this.filter,
			this.paginator.getOffset(count),
			this.paginator.pageSize);
	} else {
		var subjects = dao.listManagedSubjects(
			this.ui.session.user.id,
			this.ui.session.user.managedByRoleId,
			this.view.typeFilterValue,
			this.view.nameFilterValue,
			this.view.emailFilterValue,
			this.paginator.getOffset(count),
			this.paginator.pageSize);
	}

	return subjects;
}



SubjectOverviewController.prototype.updateTypeFilter = function(type) {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updateTypeFilter(type="  + type + ")");


	if (type == SubjectOverviewController.SELECT_ANY_VALUE) {
		delete this.filter.type;
		this.view.typeFilterValue = "";
	} else {
		this.view.typeFilterValue = type;
		var types = this.getSubjectTypes();
		for (var i = 0; i < types.length; i++) {
			if (type == types[i]) {
				this.filter.type = types[i];
				break;
			}
		}
	}
}



SubjectOverviewController.prototype.updateNameFilter = function(name) {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updateNameFilter(name=" + name + ")");

	this.view.nameFilterValue = name;

	if (name.length == 0) {
		delete this.filter.name;
	} else {
		this.filter.name = {
			compOp: "like",
			value: "%" + name.toLowerCase() + "%"
		}
	}
}



SubjectOverviewController.prototype.updateEmailFilter = function(email) {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updateEmailFilter(email=" + email + ")");

	this.view.emailFilterValue = email;

	if (email.length == 0) {
		delete this.filter.email;
	} else {
		this.filter.email = {
			compOp: "like",
			value: "%" + email.toLowerCase() + "%"
		}
	}
}
