/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2021 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model to issue a new certificate revocation list
 */

var ServiceRequestModel			= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var CRLGenerator			= require('scsh/x509/CRLGenerator').CRLGenerator;
var CRLPublisher			= require('pki-as-a-service/service/CRLPublisher').CRLPublisher;



/**
 * Data model to issue a new certificate revocation list
 *
 *
 * @constructor
 */
function IssueCRLRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
}

IssueCRLRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
IssueCRLRequestModel.constructor = IssueCRLRequestModel;

exports.IssueCRLRequestModel = IssueCRLRequestModel;



IssueCRLRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		this.form = [];

		var trustCenter = this.getRecipient();
		var caList = trustCenter.getCAList();
		var selectionModel = [];
		for (var i = 0; i < caList.length; i++) {
			var ca = caList[i];
			selectionModel.push( { id: ca.id, value: ca.name, selected: ca.id == this.model.holder } ) ;
		}

		var selectableHolder = this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT;
		var fieldset = {
			id: "crl",
			legend: "msg.crl.data",
			fields: [
				{
					id: "holder",
					label: "msg.crl.holder",
					type: "select",
					editable: selectableHolder,
					required: true,
					value: selectionModel
				}
			]
		};

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_COMPLETE) {
			fieldset.fields.push(
				{
					id: "crlNumber",
					label: "msg.crl.crlNumber",
					type: "number",
					editable: false,
					required: false,
					value: this.model.crlNumber
				});
			fieldset.fields.push(
				{
					id: "crlEntries",
					label: "msg.crl.crlEntries",
					type: "number",
					editable: false,
					required: false,
					value: this.model.crlEntries
				});
		}

		this.form.push(fieldset);
	}

	return this.form;
}



IssueCRLRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	var trustCenter = this.getRecipient();

	if (!trustCenter.isCertificationOfficer(user)) {
		return this.actionList;
	}

	if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT) {
		this.actionList.push("action.crl.select");
	}
	if (this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_COMPLETE &&
		this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_EDIT) {
		if (this.isCATokenInHSMService(trustCenter)) {
			this.actionList.push("action.crl.create.hsmservice");
		}
	} else {
		if (this.model.crl) {
			this.actionList.push("action.crl.download");
		}
	}

	return this.actionList;
}



IssueCRLRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.crl.select":
			this.setStatusInfo("CA selected");
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
			break;
		case "action.crl.create.hsmservice":
			this.issueCRL(user);
			break;
	}

	this.commit(user.id);
	return true;
}



IssueCRLRequestModel.prototype.isCATokenInHSMService = function(trustCenter) {
	var ca = trustCenter.getSigner(this.model.holder);

	var cp = trustCenter.getCryptoProvider(ca.keyDomain);
	if (cp) {
		return true;
	}

	return false;
}



IssueCRLRequestModel.prototype.issueCRL = function(user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "issueCRL()");

	var trustCenter = this.getRecipient();

	// Create CRL
	var ca = trustCenter.getX509CertificateIssuer(this.model.holder);
	var crl = ca.issueCRL();

	// Process CRL
	var crlBin = crl.getBytes();
	var crlGen = new CRLGenerator();
	this.model.crlNumber = crlGen.loadCRLEntries(crlBin);
	this.model.crlEntries = crlGen.revokedCertificates.length;
	this.model.crl = crlBin.toString(HEX);

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
	this.setStatusInfo("Certificate Revocaction List published");

	this.commit(user.id);
}



IssueCRLRequestModel.prototype.getCRL = function() {
	if (!this.model.crl) {
		throw new GPError(module.id, GPError.OBJECT_NOT_FOUND, 0, "No CRL generated by this service request");
	}

	return new ByteString(this.model.crl, HEX);
}



IssueCRLRequestModel.prototype.getFilename = function() {
	if (!this.model.crlNumber || !this.model.holder) {
		throw new GPError(module.id, GPError.OBJECT_NOT_FOUND, 0, "No CRL generated by this service request");
	}

	var holderDAO = this.service.daof.getHolderDAO();
	var holder = holderDAO.getHolderById(this.model.holder);

	var trustCenter = this.getRecipient();

	var filename = CRLPublisher.getFilename(holder, trustCenter.bo, this.model.crlNumber);
	return filename;
}
