/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2010 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Manage user authorization
 */



var ServiceRequestModel = require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;



function AuthorizationManager(service) {
	this.service = service;
}

exports.AuthorizationManager = AuthorizationManager;



AuthorizationManager.prototype.getUser = function(id) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getUser(" + id + ")");

	var tokenDAO = this.service.daof.getTokenDAO();
	var token = tokenDAO.getToken(id);
	if (token == null) {
		token = tokenDAO.newToken(id);
	}

	var user = { name: "", pinRequired: true };
	var subject;

	if (token.subjectId) {
		var subjectDAO = this.service.daof.getSubjectDAO();
		subject = subjectDAO.getSubject(token.subjectId);
	}

	if (subject && (subject.type == "Person")) {
		if (subject.name) {
			user.name = subject.name;
		} else {
			user.name = subject.email;
		}
		user.id = subject.id;
		user.tokenId = token.id;
		user.tokenPath = token.path;
		user.lang = "EN";
		if (subject.managedByRoleId) {
			user.managedByRoleId = subject.managedByRoleId;
		}

		var roleDAO = this.service.daof.getRoleDAO();
		user.roles = roleDAO.getAssignedRoleIds(user.id);
	}

	return user;
}



AuthorizationManager.prototype.loginAllowed = function(user) {
	return typeof(user.id) != "undefined" ;
}



AuthorizationManager.prototype.processBlockedToken = function(user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "processBlockedToken(" + user.id + ")");

	if (typeof(user.managedByRoleId) == "undefined") {
		GPSystem.log(GPSystem.DEBUG, module.id, "Not a managed user (" + user.id + ")");
		return;
	}

	var subjectDAO = this.service.daof.getSubjectDAO();
	var subject = subjectDAO.getSubject(user.id);

	var issueTSR = this.service.getServiceRequestById(subject.serviceRequestId);
	if (!issueTSR.isResetRetryCounterAllowed()) {
		GPSystem.log(GPSystem.DEBUG, module.id, "Reset Retry Counter is not allowed");
		return;
	}

	var srDAO = this.service.daof.getServiceRequestDAO();

	// Check for a pending service request

	var process = "ResetRetryCounterServiceRequest";
	var filter = {
		originatorId: user.id,
		process: process
	};
	var result = srDAO.listServiceRequestsByFilter(filter);
	for (var i = 0; i < result.length; i++) {
		var sr = result[i];
		if (sr.getContent().tokenId != user.tokenId) {
			continue;
		}

		if (sr.lifecycle > ServiceRequestModel.LIFECYCLE_CLOSED
			&& sr.lifecycle <= ServiceRequestModel.LIFECYCLE_REJECTED) {
			GPSystem.log(GPSystem.DEBUG, module.id, "There is already an active " + process + " for token " + user.tokenId);
			return;
		}
	}

	// Create new service request

	var factory = this.service.pluginRegistry.getFactoryForProcess(process);
	if (!factory) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Can not find factory for process " + process);
	}

	var ctrl = factory.getControllerForProcess(process);
	var parameter = {
		lang: user.lang,
		tokenId: user.tokenId,
		originatorId: user.id,
		issueTSR: issueTSR,
	};
	var sr = ctrl.createRequest(this.service, parameter);
}
