/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model for requesting a certificate from the DFN-PKI
 */

var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var KeyDomain			= require('pki-as-a-service/service/KeyDomain').KeyDomain;
var SmartCardHSM		= require('scsh/sc-hsm/SmartCardHSM').SmartCardHSM;
var HSMKeyStore			= require("scsh/sc-hsm/HSMKeyStore").HSMKeyStore;
var CryptoProvider		= require('scsh/sc-hsm/CryptoProvider').CryptoProvider;
var DNEncoder			= require("scsh/x509/DNEncoder").DNEncoder;
var PKIXCommon			= require("scsh/x509/PKIXCommon").PKIXCommon;
var X509Signer			= require('scsh/x509/X509Signer').X509Signer;
var X509Name			= require('scsh/x509/X509Name').X509Name;
var X509CertificateStore	= require('scsh/x509/X509CertificateStore').X509CertificateStore;
var Holder			= require('scsh/pki-db/Holder').Holder;
var PKCS10 			= require('scsh/pkcs/PKCS10').PKCS10;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function ResetRetryCounterServiceRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
}

ResetRetryCounterServiceRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
ResetRetryCounterServiceRequestModel.constructor = ResetRetryCounterServiceRequestModel;

exports.ResetRetryCounterServiceRequestModel = ResetRetryCounterServiceRequestModel;



ResetRetryCounterServiceRequestModel.VERIFICATION_METHOD_POST_IDENT = "Post-Ident";
ResetRetryCounterServiceRequestModel.VERIFICATION_METHOD_BASIS_KEY_USER = "Basis-KeyUser";
ResetRetryCounterServiceRequestModel.verificationMethods = [
	ResetRetryCounterServiceRequestModel.VERIFICATION_METHOD_POST_IDENT,
	ResetRetryCounterServiceRequestModel.VERIFICATION_METHOD_BASIS_KEY_USER
];



ResetRetryCounterServiceRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		var tokenDAO = this.service.daof.getTokenDAO();
		var token = tokenDAO.getTokenById(this.model.tokenId);

		this.form = [{
			id: "token",
			legend: "msg.token",
			fields: [
				{ id: "tokenpath", label: "msg.tokenpath", type: "text", editable: false, required: false, value: token.path },
		]}];

		if (typeof(this.model.pin) != "undefined") {
			if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_CLOSED) {
				var editable = false;
			} else if (this.isRegistrationOfficer(user) && (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_APPROVE)) {
				var editable = true;
			}
			this.form[0].fields.push(
				{ id: "pin", label: "msg.rrc.pin", type: "text", editable: editable, required: true, value: this.model.pin }
			);
		}
	}

	return this.form;
}



ResetRetryCounterServiceRequestModel.prototype.isRegistrationOfficer = function(user) {
// 	return this.getRecipient().isRegistrationOfficer(user);
	var raRoleId = this.getRARoleId();
	return (typeof(user.roles) != "undefined") && (user.roles.indexOf(raRoleId) >= 0);
}



ResetRetryCounterServiceRequestModel.prototype.isAssignedToRegistrationOfficer = function(user) {
	if (!this.bo.assignedToRole) {
		return false;
	}

	var raRoleId = this.getRARoleId();

	if (this.bo.assignedToRole != raRoleId) {
		return false;
	}

	return (typeof(user.roles) != "undefined") && (user.roles.indexOf(raRoleId) >= 0);
}



ResetRetryCounterServiceRequestModel.prototype.isManagerOfRequester = function(user) {
	return this.getOriginator().isManager(user);
}



ResetRetryCounterServiceRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_CLOSED) {
		return this.actionList;
	}

	if (this.isRegistrationOfficer(user)) {
		if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_APPROVE) {
			if (typeof(this.model.pin) != "undefined") {
				this.actionList.push("action.save");
			}
			this.actionList.push("action.approve");
			this.actionList.push("action.reject");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_REJECTED ||
			this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_ERROR) {
			this.actionList.push("action.close");
		}
	}

	return this.actionList;
}



ResetRetryCounterServiceRequestModel.prototype.getRARoleId = function() {
// 	return this.model.raRoleId;
	return this.getIssueTokenSR().getRecipient().getRARoleId()
}



ResetRetryCounterServiceRequestModel.prototype.handleAction = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleAction(" + user.id + "," + action + ")");

	switch(action) {
		case "action.save":
			this.handleSave();
			break;
		case "action.approve":
			var tokenDAO = this.service.daof.getTokenDAO();
			tokenDAO.addTokenAction(this.model.tokenId, this.getId());

			this.assignToRole(0);

			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_PRODUCE);
			this.setStatusInfo("Request approved");
			break;
		case "action.reject":
			this.assignToRole(0);
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_REJECTED);
			this.setStatusInfo("Request rejected");
			break;
		case "action.close":
			this.assignToRole(0);
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_CLOSED);
			this.setStatusInfo("Request closed");
			break;
	}
}



ResetRetryCounterServiceRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	this.handleAction(user, action);

	this.commit(user.id);
	return true;
}



ResetRetryCounterServiceRequestModel.prototype.handleSave = function() {
	if (this.getLifeCycle() == ServiceRequestModel.LIFECYCLE_NEW) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
	}
	this.setStatusInfo("Request data entered");
}



ResetRetryCounterServiceRequestModel.prototype.getIssueTokenSR = function() {
	return this.service.getServiceRequestById(this.model.issueTSRId);
}



ResetRetryCounterServiceRequestModel.prototype.resetRetryCounter = function(sc, chain, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "resetRetryCounter");

	var tmkkcv = sc.getTokenManagementKeyKCV();
	var salt = sc.getTokenManagementKeySalt();

	var issueTSR = this.getIssueTokenSR();
	var key = issueTSR.getDerivationKey();
	if (!key) {
		this.assignToRole(this.getRARoleId());
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_APPROVE);
		this.setStatusInfo("Token Management Key not found or not authorized");
		return false;
	}

	var kcv = issueTSR.getKCV(key);
	if (typeof(tmkkcv) != "undefined") {
		if (!kcv.equals(tmkkcv)) {
			GPSystem.log(GPSystem.DEBUG, module.id, "Given    KCV: " + kcv.toString(HEX));
			GPSystem.log(GPSystem.DEBUG, module.id, "Expected KCV: " + tmkkcv.toString(HEX));

			this.assignToRole(this.getRARoleId());
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_ERROR);
			this.setStatusInfo("Wrong Token Management Key");
			return false;
		}
	}


	var soPIN = issueTSR.deriveSOPIN();
	GPSystem.log(GPSystem.DEBUG, module.id, "soPIN: " + soPIN.toString(HEX));
	if (typeof(this.model.pin) != "undefined") {
		var pin = new ByteString(this.model.pin, ASCII);
	} else {
		var pin = null;
	}
	sc.unblockUserPIN(soPIN, pin);

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
	this.setStatusInfo("Retry Counter reset");

	return true;
}



ResetRetryCounterServiceRequestModel.prototype.handleSmartCardHSM = function(sc, chain, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleCardAction for status: " + this.bo.lifecycle);
	switch(this.bo.lifecycle) {
		case ServiceRequestModel.LIFECYCLE_PRODUCE:
			this.resetRetryCounter(sc, chain.devicecert, user);
			break;
		default:
			GPSystem.log(GPSystem.ERROR, module.id, "Unexpected handleCard in state " + this.bo.lifecycle);
			return;
	}

	if (typeof(user) == "undefined") {
		var id = this.getOriginatorId();
	} else {
		var id = user.id;
	}
	this.commit(id);
}



ResetRetryCounterServiceRequestModel.prototype.handleCard = function(card, session, reqinfo) {
	var user = session.user;

	var sc = new SmartCardHSM(card);
	var devAutCert = sc.readBinary(SmartCardHSM.C_DevAut);
	var crypto = new Crypto();
	var chain = SmartCardHSM.validateCertificateChain(crypto, devAutCert);
	if (chain == null) {
		this.setStatusInfo("Could not authenticate SmartCard-HSM");
		return;
	}

	sc.openSecureChannel(crypto, chain.publicKey);

	this.handleSmartCardHSM(sc, chain, user);
}
