/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2018 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for IssueCRLServiceRequest
 */

var CommonUI			= require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N			= require('scsh/srv-cc1/I18N').I18N;
var TrustCenter			= require('pki-as-a-service/subjects/TrustCenter').TrustCenter;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.crl.title": "Issue Certificate Revocation List",
		"msg.crl.data": "Certificate Revocation List",
		"msg.crl.holder": "CA",
		"msg.crl.crlNumber": "CRL Sequence No",
		"msg.crl.crlEntries": "Revoked Certificates",

		"action.crl.select": "Select",
		"action.crl.create.hsmservice" : "Issue",
		"action.crl.download": "Download CRL"
	}}
]);



function IssueCRLRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new IssueCRLRequestController()");
}

exports.IssueCRLRequestController = IssueCRLRequestController;



IssueCRLRequestController.prototype.createRequest = function(srct, parameter) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "IssueCRLServiceRequest",
		title: srct.i18n("msg.crl.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: parameter.recipientId,
		state: "New",
		content: {
			holderName: parameter.holderName
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



IssueCRLRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var form =
		<div>
			<h1>Issue Certificate Revocation List</h1>
		</div>

	var parameter = CommonUI.parseQueryString(req.queryString);
	var id = parameter.ca;
	var subjectLink = srct.ui.homeURL(req.url) + "/subject/" + id;

	var trustCenter = this.getTrustCenterSubject(req, res, srct);

	if (!trustCenter.isCertificationOfficer(srct.ui.session.user)) {
		form.appendChild(
			<p>You are not authorized to issue a new Certificate Revocation List.</p>
		);

		return form;
	}

	var caList = trustCenter.getCAList();
	if (caList.length == 0) {
		form.appendChild(
			<p>Currently this TrustCenter doesn't have CA to issue a certificate revocation list for.</p>
		);

		return form;
	}

	form.appendChild(
		<p>This service will issue a new Certificate Revocation List.</p>
	);

	form.appendChild(
		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
				<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
		</form>
	);

	return form;
}



IssueCRLRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	var trustCenter = this.getTrustCenterSubject(req, res, srct);

	if (trustCenter.isCertificationOfficer(srct.ui.session.user)) {
		var parameter = {recipientId: trustCenter.getId()};
		var bo = this.createRequest(srct, parameter);

		srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
		return true;
	}

	return false;
}



IssueCRLRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	return div;
}



IssueCRLRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "postProcessAction: " + action);

	switch(action) {
		case "action.crl.select":
			if (!sr.setFields(srct.ui.session.user, req.parameters)) {
				var page = srct.getProcessView(req, res, sr);
				srct.ui.sendPage(req, res, req.url, page);
				return true;
			}
		break;
		case "action.crl.download":
			this.downloadCRL(req, res, srct, sr);
			return true;
	}
	sr.perform(srct.ui.session.user, action);
	return false;
}



IssueCRLRequestController.prototype.getTrustCenterSubject = function(req, res, srct) {
	var parameter = CommonUI.parseQueryString(req.queryString);
	var ca = Number(parameter.ca);

	if (isNaN(ca) || ca == 0) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Argument ca not valid in URL");
	}

	var subject = srct.service.getSubject(ca);
	if (!(subject instanceof TrustCenter)) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "The given subject is not a trust center");
	}

	return subject;
}



IssueCRLRequestController.prototype.downloadCRL= function(req, res, srct, sr) {
	GPSystem.log(GPSystem.DEBUG, module.id, "downloadCRL()");

	var filename = sr.getFilename();
	var crl = sr.getCRL();

	res.setContentType("application/octet-stream");
	res.setContentLength(crl.length);
	res.addHeaderField("Content-Disposition", "attachment; filename=\"" + filename + "\"");
	res.write(crl);
}
