/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A view for service requests
 */

var CommonUI			= require('scsh/srv-cc1/CommonUI').CommonUI;
var Paginator			= require('pki-as-a-service/ui/Paginator').Paginator;
var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;



function ServiceRequestOverview(ui, selector) {
	this.ui = ui;
	this.service = ui.service;
	this.selector = selector;

	this.paginator = new Paginator();

	this.filter = {};
	this.processFilterValue = "";
	this.detailsFilterValue = "";
	this.lifeCycleFilterValue = -1;
}

exports.ServiceRequestOverview = ServiceRequestOverview;



ServiceRequestOverview.SELECT_ANY_VALUE = "Any";



ServiceRequestOverview.prototype.handleRequest = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleRequest()");
	GPSystem.log(GPSystem.DEBUG, module.id, "req.method:" + req.method);
	GPSystem.log(GPSystem.DEBUG, module.id, "req.queryString:" + req.queryString);

	if (req.queryString) {
		this.handleOperation(req, res);
	}

	if (req.url.length != 2) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Invalid URL");
	}

	if (req.parameters) {
		this.handlePostOperation(req, res);
	}

	var page = this.getPage(req, res);
	this.ui.sendPage(req, res, req.url, page);
}



ServiceRequestOverview.prototype.handleOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleOperation(" + req.queryString + ")");
	var operation = CommonUI.parseQueryString(req.queryString);

	switch(operation.op) {
		case "changepagesize":
			var val = parseInt(operation.pagesize);
			if (val > 0) {

				this.paginator.pageSize = val;
			}
			break;
		case "changeoffset":
			var val = parseInt(operation.offset);
			this.paginator.offset = val;
			break;
	}

	var url = this.ui.homeURL(req.url) + "/sroverview";
	res.setStatus(HttpResponse.SC_SEE_OTHER);
	res.addHeader("Location", url);
}



ServiceRequestOverview.prototype.handlePostOperation = function(req, res) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handlePostOperation( op=" + req.parameters.op + " )");
	if (req.parameters.op == "filter") {
		if (req.parameters.process) {
			this.processFilterValue = req.parameters.process;
		}

		if (typeof(req.parameters.details) != "undefined") {
			this.detailsFilterValue = req.parameters.details;
		}

		if (req.parameters.lifeCycle) {
			this.lifeCycleFilterValue = req.parameters.lifeCycle;
		}
	}

	this.updateFilter();
}



ServiceRequestOverview.prototype.updateFilter = function() {
	GPSystem.log(GPSystem.DEBUG, module.id,
		"updateFilter(process="  + this.processFilterValue
		+ ", details=" + this.detailsFilterValue
		+ ", lifeCycle=" + this.lifeCycleFilterValue + ")");

	if (this.processFilterValue == ServiceRequestOverview.SELECT_ANY_VALUE) {
		delete this.filter.process;
	} else {
// 		this.filter.process = this.processFilterValue;

		var processes = this.service.pluginRegistry.enumerateRunnableProcesses(this.ui.session.user);
		for (var i = 0; i < processes.length; i++) {
			if (this.processFilterValue == processes[i].label) {
				this.filter.process = processes[i].process;
				break;
			}
		}
	}

	if (this.detailsFilterValue.length == 0) {
		delete this.filter.details;
	} else {
		this.filter.details = {
			compOp: "like",
			value: "%" + this.detailsFilterValue.toLowerCase() + "%"
		}
	}

	if (this.lifeCycleFilterValue == ServiceRequestOverview.SELECT_ANY_VALUE) {
		delete this.filter.lifeCycle;
	} else {
		for (var i = 0; i < ServiceRequestModel.LIFECYCLE_STATES.length; i++) {
			var lifeCycle = ServiceRequestModel.LIFECYCLE_STATES[i];
			if (this.lifeCycleFilterValue == lifeCycle) {
				this.filter.lifeCycle = i;
				break;
			}
		}
	}
}



ServiceRequestOverview.prototype.getPage = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var srDAO = this.service.daof.getServiceRequestDAO();
	var records = srDAO.countServiceRequests(this.ui.session.user.id, this.ui.session.user.roles, this.filter, this.paginator.offset, this.paginator.pageSize);
	var list = srDAO.listServiceRequests(this.ui.session.user.id, this.ui.session.user.roles, this.filter, this.paginator.getOffset(records), this.paginator.pageSize);

	var page = this.renderServiceRequests(records, list, req, res);

	var t = this.ui.generateTemplate(req.url);
 	var c = t..div.(@id == "content");
 	c.div = page;

	return t;
}


ServiceRequestOverview.prototype.getProcessFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var processFilter = <form class="pure-form pure-form-stacked" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<fieldset>
			<input name="op" type="hidden" value="filter"/>
			<div class="pure-g">
				<div class="pure-u-1 pure-u-md-1-3">
					<label for="processfilter">Process</label>
					<select id="processfilter" name="process" class="pure-u-23-24"/>
					<noscript><button class="pure-button" type="submit">Filter</button></noscript>
				</div>
			</div>
		</fieldset>
	</form>

	var selectProcess = processFilter..select.(@id == "processfilter");

	var option = <option>{ServiceRequestOverview.SELECT_ANY_VALUE}</option>;
	if (this.processFilterValue == ServiceRequestOverview.SELECT_ANY_VALUE) {
		option.@selected = "selected";
	}
	selectProcess.appendChild(option);

	var processes = this.service.pluginRegistry.enumerateRunnableProcesses(this.ui.session.user);
	for (var i = 0; i < processes.length; i++) {
		var option = <option>{processes[i].label}</option>;
// 		var option = <option>{processes[i].process}</option>;
		if (this.processFilterValue == processes[i].label) {
// 		if (this.processFilterValue == processes[i].process) {
			option.@selected = "selected";
		}
		selectProcess.appendChild(option);
	}

	return processFilter;
}



ServiceRequestOverview.prototype.getDetailsFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";


	var detailsFilter = <form class="pure-form pure-form-stacked" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<fieldset>
			<input name="op" type="hidden" value="filter"/>
			<div class="pure-g">
				<div class="pure-u-1 pure-u-md-1-3">
					<label for="detailsFilter">Details</label>
					<input id="detailsFilter" type="text" placeholder="Search for Details" class="pure-u-23-24" name="details" value={this.detailsFilterValue}/>
					<noscript><button class="pure-button" type="submit">Filter</button></noscript>
				</div>
			</div>
		</fieldset>
	</form>

	return detailsFilter;
}



ServiceRequestOverview.prototype.getLifeCycleFilter = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var lifeCycleFilter = <form class="pure-form pure-form-stacked" action="" method="post" enctype="multipart/form-data" onchange="submit()">
		<fieldset>
			<input name="op" type="hidden" value="filter"/>
			<div class="pure-g">
				<div class="pure-u-1 pure-u-md-1-3">
					<label for="lifeCycleFilter">Life-Cycle</label>
					<select id="lifeCycleFilter" name="lifeCycle" class="pure-u-23-24"/>
					<noscript><button class="pure-button" type="submit">Filter</button></noscript>
				</div>
			</div>
		</fieldset>
	</form>

	var selectLifeCycle = lifeCycleFilter..select.(@id == "lifeCycleFilter");

	var option = <option>{ServiceRequestOverview.SELECT_ANY_VALUE}</option>;
	if (this.lifeCycleFilterValue == ServiceRequestOverview.SELECT_ANY_VALUE) {
		option.@selected = "selected";
	}
	selectLifeCycle.appendChild(option);

	for (var i = 0; i < ServiceRequestModel.LIFECYCLE_STATES.length; i++) {
		var lifeCycle = ServiceRequestModel.LIFECYCLE_STATES[i];
		var option = <option>{lifeCycle}</option>;
		if (this.lifeCycleFilterValue == lifeCycle) {
			option.@selected = "selected";
		}
		selectLifeCycle.appendChild(option);
	}

	return lifeCycleFilter;
}



ServiceRequestOverview.prototype.getPaginationDiv = function() {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var pagination =
		<div class="table-control">
			<div class="">
				<form class="pure-form" action="" method="get" onchange="submit()">
					<input name="op" type="hidden" value="changepagesize"/>
					<fieldset>
						<label for="size">Table Size:</label>
						<select id="size" name="pagesize"/>
						<noscript><button class="pure-button" type="submit">Change</button></noscript>
					</fieldset>
				</form>
			</div>
		</div>;

	var selectSize = pagination..select.(@id == "size");

	var option = <option>5</option>;
	if (this.paginator.pageSize == 5) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>10</option>;
	if (this.paginator.pageSize == 10) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>20</option>;
	if (this.paginator.pageSize == 20) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>50</option>;
	if (this.paginator.pageSize == 50) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	var option = <option>100</option>;
	if (this.paginator.pageSize == 100) {
		option.@selected = "selected";
	}
	selectSize.appendChild(option);

	return pagination;
}



ServiceRequestOverview.prototype.renderServiceRequests = function(records, srlist, req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div class="card"/>;

	// Create page navigation

	var pagination = this.getPaginationDiv();

	var url = this.ui.homeURL(req.url) + "/sroverview";
	var pagenavigator = this.paginator.getPagination(records, url + "?op=changeoffset&");

	// Create table header

	var processFilter = this.getProcessFilter();
	var detailsFilter = this.getDetailsFilter();
	var lifeCycleFilter = this.getLifeCycleFilter();

	var table =
		<table class="pure-table pure-table-striped" style="table-layout:fixed">
			<thead>
				<tr>
					<th>ID</th>
					<th>{processFilter}</th>
					<th>Originator</th>
					<th>Recipient</th>
					<th>{detailsFilter}</th>
					<th>Assigned to Role</th>
					<th>{lifeCycleFilter}</th>
				</tr>
			</thead>
			<tbody/>
		</table>;

	var url = this.ui.homeURL(req.url);

	// Fill table

	var roleDAO = this.service.daof.getRoleDAO();
	for (var i = 0; i < srlist.length; i++) {
		var sr = srlist[i];
		var originatorSubject = this.service.getSubject(sr.originatorId);
		if (originatorSubject.getName()) {
			var originator = originatorSubject.getName();
		} else {
			var originator = originatorSubject.getEmail();
		}

		if (sr.recipientId) {
			var recipientSubject = this.service.getSubject(sr.recipientId);
			if (recipientSubject.getName()) {
				var recipient = <a href={url + "/subject/" + sr.recipientId}>{recipientSubject.getName()}</a>;
			} else {
				var recipient = <a href={url + "/subject/" + sr.recipientId}>{recipientSubject.getEmail()}</a>;
			}
		} else {
			var recipient = "";
		}

		var details = sr.details
		if (!details) {
			var details = ""
		}

		if (sr.assignedToRole) {
			var role = roleDAO.getRoleById(sr.assignedToRole).name;
		} else {
			var role = "";
		}

		var lifeCycle = ServiceRequestModel.getLifeCycleStringForId(sr.lifecycle);

		var tr = <tr>
				<td><a href={url + "/sr/" + sr.id}>{sr.id}</a></td>
				<td>{sr.process}</td>
				<td><a href={url + "/subject/" + sr.originatorId}>{originator}</a></td>
				<td>{recipient}</td>
				<td>{details}</td>
				<td>{role}</td>
				<td>{lifeCycle}</td>
			</tr>
 		table.tbody.appendChild(tr);
	}

	var tableDiv = <div class="pure-u-1">{table}</div>;
	pagination.appendChild(tableDiv);
	pagination.appendChild(pagenavigator);
	div.appendChild(pagination);
	return div;
}
