/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2022 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for CreateTrustCenterServiceRequest
 */

var CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.estsubject.title": "Create EST Server",

		"msg.estsubject.subjectData": "EST Server Data",
		"msg.estsubject.name": "Unique Name",
		"msg.estsubject.lc": "Operation Mode",


		"action.estsubject.changeMode": "Change Operation Mode",
		"action.estsubject.changeMode.tooltip": "Change the operation mode of the EST Server",

	}}
]);



function CreateESTServerRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new CreateESTServerRequestController()");
}

exports.CreateESTServerRequestController = CreateESTServerRequestController;



CreateESTServerRequestController.prototype.getSubject = function(req, res, srct) {
	var parameter = CommonUI.parseQueryString(req.queryString);
	var ca = Number(parameter.ca);

	if (isNaN(ca) || ca == 0) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Argument ca not valid in URL");
	}

	var subject = srct.service.getSubject(ca);

	if (!(subject instanceof TrustCenter)) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Subject is not a TrustCenter instance");
	}

	return subject;
}



CreateESTServerRequestController.prototype.createRequest = function(srct, trustCenterId) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "CreateESTServerServiceRequest",
		title: srct.i18n("msg.estsubject.title"),
		originatorId: srct.ui.session.user.id,
		state: "Created",
		content: {
			trustCenter: trustCenterId
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



CreateESTServerRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	if (srct.ui.session.user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
		var parameter = CommonUI.parseQueryString(req.queryString);

		var trustCenter = this.getSubject(req, res, srct);

		if (!trustCenter.isManager(srct.ui.session.user)) {
			return false;
		}

		var bo = this.createRequest(srct, trustCenter.getId());

		srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
		return true;
	}

	return false;
}



CreateESTServerRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.submit":
	case "action.save":
	case "action.estsubject.changeMode":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



CreateESTServerRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var form =	<div>
				<h1>Create an EST-Server Subject</h1>

				<p>An EST-Server enables the Enrollment over Secure Transport (EST) for participating TrustCenter.</p>

				<p>The TrustCenter that is creating the EST-Server will be the managing TrustCenter. Anyone who is
				assigned to the TrustCenter's manager role can update the operation mode (private, public, suspended)
				of the EST-Server.</p>

				<p>The configuration of an EST-Server (i.e. trusted certificates, signer that can reenroll without RA approval)
				is TrustCenter dependent and will be done in a separate "Join EST-Server" service request.</p>

				<p>An EST-Server can be either private or public. A public EST-Server can be joined
				   by any TrustCenter in the PKI-as-a-Service Portal.
				   A private EST-Server can only be joined by managing TrustCenter.</p>
			</div>

	var trustCenter = this.getSubject(req, res, srct);

	if (trustCenter.isManager(srct.ui.session.user)) {
		form.appendChild(
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		);
	} else {
		form.appendChild(
			<p>You need the manager role to create an EST-Server for this Trust Center.</p>
		)
	}

	return form;
}



CreateESTServerRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	return div;
}
