/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for CertificateServiceRequest
 */

var CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
var SubjectDAO = require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.cr.title": "Request Certificate",
		"msg.cr.applicant": "Certificate Holder Information",
		"msg.cr.approval": "Registration Officer Review",
		"msg.cr.commonName": "Name",
		"msg.cr.reviewRemark": "Remark",
		"msg.cr.csr": "Certificate Request",
		"msg.cr.dn": "Distinguished Name",
		"msg.cr.token": "Token",
		"msg.cr.revocation": "Certificate Revocation",
		"msg.cr.revocationReason": "Reason",
		"action.cr.reqgen.usertoken": "Generate Request on Token",
		"action.cr.publish.usertoken": "Store Certificate on Token",
		"action.cr.produce.hsmservice": "Issue Certificate",
		"action.cr.revoke": "Revoke Certificate",
		"action.cr.updateStatus": "Update Revocation Status"
	}}
]);



function CertificateRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new CertificateRequestController()");
}

exports.CertificateRequestController = CertificateRequestController;





CertificateRequestController.prototype.createRequest = function(srct, parameter) {

	var holderId = parseInt(parameter.ca);
	GPSystem.log(GPSystem.DEBUG, module.id, "Selected CA:" + holderId);

	// Obtain initialized subject object
	var holderDAO = srct.ui.service.daof.getHolderDAO();
	var ca = holderDAO.getHolderById(holderId);

	if (ca == null) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Can't find CA holder");
	}

	if (!ca.certId) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Invalid CA holder");
	}

	var trustCenter = srct.service.getSubject(ca.subjectId);

	if (trustCenter == null) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Can't find trust center");
	}

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "CertificateServiceRequest",
		title: srct.i18n("msg.cr.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: trustCenter.bo.id,
		state: "New request",
		content: {
			issuer: holderId
		}
	};

	if (!trustCenter.checkServiceRequestCreation(srct.ui.session.user, t)) {
		return undefined;
	}

	if (trustCenter.canRead(srct.ui.session.user) || trustCenter.isManagerOf(srct.ui.session.user)) {
		var bo = srDAO.newServiceRequest(t);
	} else {
		return undefined;
	}

	return bo;
}



CertificateRequestController.prototype.createInitializedRequest = function(service, template) {
	var srDAO = service.daof.getServiceRequestDAO();

	var t = {
		parentServiceRequestId: template.parent,
		process: "CertificateServiceRequest",
		title: I18N.t(template.lang, "msg.cr.title"),
		details: template.commonName,
		originatorId: template.originatorId,
		recipientId: template.recipientId,
		state: "Submit",
		lifecycle: '3',
		assignedToRole: template.assignedToRole,
		content: {
			commonName: template.commonName,
			role: template.assignedToRole,
			issuer: template.issuer
		}
	};

	var bo = srDAO.newServiceRequest(t);
	var sr = service.getServiceRequestById(bo.id);

	sr.determineSubjectDN();
	sr.commit();

	return bo;
}



CertificateRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	var bo = this.createRequest(srct, req.parameters);

	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



CertificateRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.save":
	case "action.submit":
	case "action.approve":
	case "action.reject":
	case "action.cr.revoke":
	case "action.cr.updateStatus":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	case "action.cr.reqgen.usertoken":
	case "action.cr.publish.usertoken":
		srct.ui.wizard = new CardActionWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



CertificateRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div =	<div>
				<h1>Request a Certificate</h1>
				<p>This service will request a new certificate from one of the registered
				   certification authorities.</p>
			</div>

	var form =
		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<p>Please select a certification authority from which you would like to receive a certificate.</p>
			<select name="ca"/>
			<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
		</form>

	var subjectDAO = srct.ui.service.daof.getSubjectDAO();
	var l = form.select;
	var trustCenters = subjectDAO.listSubjectsByType(SubjectDAO.TYPE_TRUST_CENTER);
	for each (var subject in trustCenters) {
		var trustCenter = srct.service.getSubject(subject.id);
		if (trustCenter.canRead(srct.ui.session.user) || trustCenter.isManagerOf(srct.ui.session.user)) {
			var optgroup = <optgroup label={ trustCenter.getName() } />;
			var caList = trustCenter.getCAList();
			for each (var ca in caList) {
				optgroup.appendChild(<option value={ca.id}>{ca.getReadablePath()}</option>);
			}
			if (caList.length > 0) {
				l[0].appendChild(optgroup);
			}
		}
	}

	if (l.optgroup.length() > 0) {
		div.appendChild(<p>The service request consists of the following steps:</p>);
		div.appendChild(
			<ol>
				<li>You select from which certification authority you would like to receive a certificate</li>
				<li>You provide informations required by the certification authority</li>
				<li>A key pair is generated on your SmartCard-HSM</li>
				<li>A certificate request is generated and forwarded to a registration officer of the CA</li>
				<li>The registration officer either approves or rejects your request</li>
				<li>If the request is approved, then it is forwarded to the certification officer of the CA</li>
				<li>The certification officer will issue the certificate</li>
				<li>In the last step the certificate is stored on your SmartCard-HSM</li>
			</ol>);
		div.appendChild(form);
	} else {
		div.appendChild(<p>There is no accessible certification authority. This is either the case if there are no operational certification authorities or you are not authorized to access a private certification authority.</p>);
	}
	return div;
}



CertificateRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	if (sr.hasCertificates() > 0) {
		div.appendChild( <br/> );
		div.appendChild( <h2>Certificates</h2> );
		div.appendChild(srct.renderCertificateList(req, res, sr));
	}

	return div;
}
