/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Base Service request controller enrolling entities in a trust center
 */

var InitializationWizard = require('pki-as-a-service/ui/InitializationWizard').InitializationWizard;
var CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;
var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;



function EnrollBaseRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new EnrollBaseRequestController()");
}

exports.EnrollBaseRequestController = EnrollBaseRequestController;



/**
 * Determine the trust center for which this request shall be started
 *
 * @param {HttpRequest} req the request object
 * @param {HttpResponse} res the response object
 * @param {ServiceRequestController} srct the service request controller in which this controller is embedded.
 * @type TrustCenter
 * @return the associated TrustCenter instance
 */
EnrollBaseRequestController.prototype.getTrustCenter = function(req, res, srct) {
	var parameter = CommonUI.parseQueryString(req.queryString);
	var ca = Number(parameter.ca);

	if (isNaN(ca) || ca == 0) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Argument ca not valid in URL");
	}

	var subject = srct.service.getSubject(ca);

	if (!(subject instanceof TrustCenter)) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Subject is not a TrustCenter instance");
	}

	return subject;
}



/**
 * Handle POST request to create service request and initiate a GET after POST redirect to
 * the newly created service request.
 *
 * @param {HttpRequest} req the request object
 * @param {HttpResponse} res the response object
 * @param {ServiceRequestController} srct the service request controller in which this controller is embedded.
 * @param {String} action from the POST parameter
 * @type boolean
 * @return true if processing completed, false if it failed
 */
EnrollBaseRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	var ca = this.getTrustCenter(req, res, srct);
	if (!ca.canRead(srct.ui.session.user)) {
		return false;
	}
	if (!ca.canEnroll(srct.ui.session.user)) {
		return false;
	}

	var bo = this.createRequest(srct, ca.getId());

	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



/**
 * Handle POST request to update service request.
 *
 * @param {HttpRequest} req the request object
 * @param {HttpResponse} res the response object
 * @param {ServiceRequestController} srct the service request controller in which this controller is embedded.
 * @param {ServiceRequestModel} sr the service request model, typically a derived class
 * @param {String} action from the POST parameter
 * @type boolean
 * @return true if processing results in a new URL or an error page was generated, false if GET after POST in calling
 * service request controller is desired.
 */
EnrollBaseRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleAction: " + action);

	switch(action) {
	case "action.create":
	case "action.save":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	case "action.registerToken":
		break;
	case "action.deregisterToken":
		break;
	case "action.initialize":
		srct.ui.wizard = new InitializationWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.wizard.requestPIN = -1;
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	case "action.selectToken":
		srct.ui.wizard = new CardActionWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.wizard.requestPIN = -1;
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



/**
 * Handle GET request to show service request details
 *
 * @param {HttpRequest} req the request object
 * @param {HttpResponse} res the response object
 * @param {ServiceRequestController} srct the service request controller in which this controller is embedded.
 * @param {ServiceRequestModel} sr the service request model, typically a derived class
 * @type XML
 * @return The XHTML with service request details to be embedded in the service request controller frame.
 */
EnrollBaseRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	return div;
}
