/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2022 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service Request model to import and trust certificates from
 * a PKCS12 keystore
 */



var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var PKIXCommon			= require('scsh/x509/PKIXCommon').PKIXCommon;
var X509CertificateStore	= require('scsh/x509/X509CertificateStore').X509CertificateStore;



/**
 * Data model for importing trusted certificates
 *
 *
 * @constructor
 */
function ImportTrustedCertificatesRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
	this.validationResult = [];
}

ImportTrustedCertificatesRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
ImportTrustedCertificatesRequestModel.constructor = ImportTrustedCertificatesRequestModel;

exports.ImportTrustedCertificatesRequestModel = ImportTrustedCertificatesRequestModel;



ImportTrustedCertificatesRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {

		this.form = [{
			id: "applicant",
			legend: "msg.itc.container",
			fields: []
		}];

		if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT) {
			this.form[0].fields.push(
				{ id: "password", label: "msg.itc.pw", type: "text", required: false, editable: true, value: "" },
				{ id: "pkcs12", label: "msg.itc.pkcs12", type: "file", required: this.model.pkcs12 ? false : true, editable: true, value: this.model.pkcs12 ? "uploaded-pkcs12" : "nothing" }
			);

		} else {
			if (this.model.aliases != null) {
				var certlist = [];
				for (var i = 0; i < this.model.aliases.length; i++) {
					var alias = this.model.aliases[i];
					var e = { id: i, value: alias };
					if (this.model.selectedCertificates != null && this.model.selectedCertificates.indexOf(i) >= 0) {
						e.selected = true;
					}
					certlist.push(e);
				}

				var editable = this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE;
				this.form[0].fields.push(
					{ id: "selectedCertificates", label: "msg.itc.certificates", type: "select", multiselect: true, editable: editable, value: certlist }
				);
			}
		}
	}

	return this.form;
}



ImportTrustedCertificatesRequestModel.prototype.updateModel = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "updateModel(" + fld.id + "," + val + ")");

	switch(fld.id) {
		case "password":
			GPSystem.log(GPSystem.DEBUG, module.id, "Received PW " + val);
			// Don't save password in model
			// It can be read via rest
			this.password = val;
			return true;
		case "selectedCertificates":
			break;
		case "pkcs12":
			break;
		default:
			return false;
	}

	this.model[fld.id] = val;
	return true;
}




ImportTrustedCertificatesRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];


	var trustCenter = this.getRecipient();
	var raOfficer = trustCenter.isRegistrationOfficer(user);

	if ((user.id == this.getOriginatorId()) || raOfficer) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW ||
			this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			this.actionList.push("action.itc.upload");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) {
			this.actionList.push("action.itc.import");
			this.actionList.push("action.itc.discard");
		}
	}

	return this.actionList;
}



ImportTrustedCertificatesRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.itc.upload":
			this.handleUpload();
			break;
		case "action.itc.discard":
			this.model.aliases = [];
			this.model.certificates = {};
			this.model.selectedCertificates = [];
			this.setStatusInfo("PKCS12 container discarded");
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
			break;
		case "action.itc.import":
			this.importCertificates();
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
			this.setStatusInfo("Certificate imported");
			break;
	}

	this.commit(user.id);
	return true;
}



ImportTrustedCertificatesRequestModel.prototype.handleUpload = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleUpload()");

	if (this.model.pkcs12 != null) {
		// TODO Unique Filename
		var fn = "temp_uploaded_truststore.p12";
		PKIXCommon.writeFileToDisk(fn, this.model.pkcs12)
		var absFilename = GPSystem.mapFilename(fn);
		GPSystem.log(GPSystem.DEBUG, module.id, "uploaded truststore " + absFilename);

		GPSystem.log(GPSystem.DEBUG, module.id, "Using PW " + this.password);

		var p12 = new KeyStore("BC", "PKCS12", absFilename, this.password);

		this.certlist = [];

		var aliases = p12.getAliases();
		this.model.aliases = aliases;
		this.model.certificates = {};
		GPSystem.log(GPSystem.DEBUG, module.id, "aliases " + aliases);
		for (var i = 0; i < aliases.length; i++) {
			var alias = aliases[i];
			var cert = p12.getCertificate(alias);
			GPSystem.log(GPSystem.DEBUG, module.id, alias + " cert: " + cert);
			this.model.certificates[alias] = cert.getBytes();
		}

		// TODO delete truststore
	} else {
		GPSystem.log(GPSystem.DEBUG, module.id, "Missing pkcs12 file");
	}


	this.setStatusInfo("PKCS12 container uploaded");
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_APPROVE);
}



/*
 * - Only import if there is a full cert chain.
 * - Make current only if there is no other certificate
 * - Start with root cert and build up holder path
 * - TODO Show certificates by subject dn
 */
ImportTrustedCertificatesRequestModel.prototype.importCertificates = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "importCertificates()");
	var holderDAO = this.service.daof.getHolderDAO();
	var certDAO = this.service.daof.getCertificateDAO();

	var store = new X509CertificateStore(this.service.daof);

	var certificateMap = {};
	var toBeProcessed = [];

	var trustCenter = this.getRecipient();
	var content = trustCenter.bo.getContent();
	if (!content.trustedCA) {
		content.trustedCA = [];
	}
	var trustedCA = content.trustedCA;

	// First process all root certificates

	for (var i = 0; i < this.model.selectedCertificates.length; i++) {
		// Is certificate already imported?
		var aliasIndex = this.model.selectedCertificates[i];
		var alias = this.model.aliases[aliasIndex];
		var certBin = this.model.certificates[alias];
		var x509 = new X509(certBin);

		if (x509.getIssuerDNString().equals(x509.getSubjectDNString())) {
			// Process root certifcate
			var path = "/" + x509.getSubjectDNString();
			var certDTO = store.storeCertificate(path, x509, false, this.getId());
			trustedCA.push(certDTO.id);
			certificateMap[x509.getSubjectDNString()] = x509;
		} else {
			toBeProcessed.push(x509);
		}
	}

	// Find subordinate certificates
	GPSystem.log(GPSystem.DEBUG, module.id, "Find subordinate certificates");
	do {
		var count = toBeProcessed.length;
		GPSystem.log(GPSystem.DEBUG, module.id, "#toBeProcessed " + count);
		for (var i = 0; i < toBeProcessed.length; i++) {
			var cert = toBeProcessed[i];
			GPSystem.log(GPSystem.DEBUG, module.id, "Process " + cert.getSubjectDNString());
			var parent = certificateMap[cert.getIssuerDNString()];
			if (parent != null) {
				GPSystem.log(GPSystem.DEBUG, module.id, "Found subordinate certificate " + parent.getSubjectDNString());
				var path = "/" + parent.getSubjectDNString()
					+ "/" + cert.getSubjectDNString();
				var certDTO = store.storeCertificate(path, cert, false, this.getId());
				trustedCA.push(certDTO.id);
				certificateMap[cert.getSubjectDNString()] = cert;
				toBeProcessed.splice(i, 1);
				break;
			}
		}
	} while (toBeProcessed.length > 0 &&
		toBeProcessed.length != count);

	for (var i = 0; i < toBeProcessed.length; i++) {
		var cert = toBeProcessed[i];
		GPSystem.log(GPSystem.DEBUG, module.id, "Could not build certificate chain for certificate " + cert);
	}

	// Persist trustedCA
	var subjectDAO = this.service.daof.getSubjectDAO();
	subjectDAO.updateContent(trustCenter.bo);
}
