/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2022 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for CreateTrustCenterServiceRequest
 */

var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var ServiceRequestModel = require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.itc.title": "Import Trusted Certificates",

		"msg.itc.container": "PKCS12 Keystore Data",
		"msg.itc.pw": "PKCS12 Password",
		"msg.itc.pkcs12": "Keystore",
		"msg.itc.certificates": "Certificates",

		"action.itc.upload": "Upload",
		"action.itc.discard": "Discard",
		"action.itc.import": "Import",
	}}
]);



function ImportTrustedCertificatesRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new ImportTrustedCertificatesRequestController()");
}

exports.ImportTrustedCertificatesRequestController = ImportTrustedCertificatesRequestController;



ImportTrustedCertificatesRequestController.prototype.getSubject = function(req, res, srct) {
	var parameter = CommonUI.parseQueryString(req.queryString);
	var ca = Number(parameter.ca);

	if (isNaN(ca) || ca == 0) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Argument ca not valid in URL");
	}

	var subject = srct.service.getSubject(ca);

	if (!(subject instanceof TrustCenter)) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Subject is not a TrustCenter instance");
	}

	return subject;
}



ImportTrustedCertificatesRequestController.prototype.createRequest = function(srct, trustCenterId) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "ImportTrustedCertificatesServiceRequest",
		title: srct.i18n("msg.itc.title"),
		originatorId: srct.ui.session.user.id,
		recipientId: trustCenterId,
		state: "Created",
		content: {
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



ImportTrustedCertificatesRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	if (srct.ui.session.user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
		var parameter = CommonUI.parseQueryString(req.queryString);

		var trustCenter = this.getSubject(req, res, srct);

		if (!trustCenter.isRegistrationOfficer(srct.ui.session.user)) {
			return false;
		}

		var bo = this.createRequest(srct, trustCenter.getId());

		srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
		return true;
	}

	return false;
}



ImportTrustedCertificatesRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.submit":
	case "action.save":
	case "action.itc.upload":
	case "action.itc.import":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



ImportTrustedCertificatesRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var form =	<div>
				<h1>Import Trusted Certificates</h1>
				<p>This service request lets you import trusted certificates from a PKCS#12 encoded keystore.</p>
				<p>The imported certificates will be trusted by the associated TrustCenter.</p>
				<p>Only certifcates with a verified certificate chain will be imported.</p>
			</div>

	var trustCenter = this.getSubject(req, res, srct);

	if (trustCenter.isRegistrationOfficer(srct.ui.session.user)) {
		form.appendChild(
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		);
	} else {
		form.appendChild(
			<p>You need the registration officer (RA) role to continue.</p>
		)
	}

	return form;
}



ImportTrustedCertificatesRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	// TODO This will only show certificates that were imported
	// in the database by this service request (cert.srId == sr.id)
	// Refactor to link all trusted certificates including
	// certificates which were already imported by another service request
	if (sr.getLifeCycle() >= ServiceRequestModel.LIFECYCLE_INUSE) {
		div.appendChild( <br/> );
		div.appendChild( <h2>Certificates</h2> );
		div.appendChild(srct.renderCertificateList(req, res, sr));
	}

	return div;
}
