/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model for requesting a certificate for a key on a SmartCard-HSM
 */

var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var CVC				= require('scsh/eac/CVC').CVC;
var SmartCardHSM		= require('scsh/sc-hsm/SmartCardHSM').SmartCardHSM;
var HSMKeyStore			= require("scsh/sc-hsm/HSMKeyStore").HSMKeyStore;
var CryptoProvider		= require('scsh/sc-hsm/CryptoProvider').CryptoProvider;
var DNEncoder			= require("scsh/x509/DNEncoder").DNEncoder;
var PKIXCommon			= require("scsh/x509/PKIXCommon").PKIXCommon;
var Holder			= require('scsh/pki-db/Holder').Holder;
var X509Signer			= require('scsh/x509/X509Signer').X509Signer;
var KeyDomain			= require('pki-as-a-service/service/KeyDomain').KeyDomain;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function CertificateRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
}

CertificateRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
CertificateRequestModel.constructor = CertificateRequestModel;

exports.CertificateRequestModel = CertificateRequestModel;



CertificateRequestModel.prototype.getTokenList = function() {
	var tokenDAO = this.service.daof.getTokenDAO();
	var tokens = tokenDAO.getTokenListBySubjectId(this.getOriginatorId());
	return tokens;
}



CertificateRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		var editable = ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) || (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT)) && (user.id == this.getOriginatorId());

		this.form = [{
			id: "applicant",
			legend: "msg.cr.applicant",
			fields: [
				{ id: "commonName", label: "msg.cr.commonName", type: "text", size: 64, required: true, editable: editable, value: (this.model.commonName ? this.model.commonName : "") }
			]}
		];

		// Request Token
		if (this.model.tokenPath) {
			this.form[0].fields.push(
				{ id: "tokenPath", label: "msg.cr.token", type: "text", size: 100, required: false, editable: false, value: this.model.tokenPath }
			);
		}

		if (this.hasRequest()) {
			var req = this.getRequest();
			this.form[0].fields.push(
				{ id: "csr", label: "msg.cr.csr", type: "text", size: 100, required: false, editable: false, value: req.toString() }
			);
		}

		if (this.model.subjectDN || this.hasRequest()) {
			if (this.model.subjectDN) {
				var dnval = PKIXCommon.dnToString(this.model.subjectDN);
			} else {
				var dnval = PKIXCommon.dnToString(this.transformSubjectDN());
			}

			var editableDN = this.isRegistrationOfficer(user) && this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_PRODUCE;
			this.form[0].fields.push(
				{ id: "dn", label: "msg.cr.dn", type: "text", size: 100, required: false, editable: editableDN, value: dnval }
			);
		}

		if (this.model.reviewRemark || (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE)) {
			var editable = (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) && this.isRegistrationOfficer(user);
			this.form.push(
			{
				id: "approval",
				legend: "msg.cr.approval",
				fields: [
					{ id: "reviewRemark", label: "msg.cr.reviewRemark", type: "textarea", size: 4096, rows:10, cols:80, editable: editable, value: (this.model.reviewRemark ? this.model.reviewRemark : "") }
				]
			});
		}

		if (this.bo.lifecycle > ServiceRequestModel.LIFECYCLE_DELIVER &&
			(this.isRegistrationOfficer(user) || this.isManagerOfRequester(user))) {
			var reasons = [];
			reasons.push( { id: OCSPQuery.REVOKED, value: "Undefined Reason", selected: ( OCSPQuery.REVOKED == this.model.revocationReason) } );
			reasons.push( { id: OCSPQuery.KEYCOMPROMISE, value: "Key Compromised", selected: ( OCSPQuery.KEYCOMPROMISE == this.model.revocationReason) } );
			reasons.push( { id: OCSPQuery.CACOMPROMISE, value: "CA Key Compromised", selected: ( OCSPQuery.CACOMPROMISE == this.model.revocationReason) } );
			reasons.push( { id: OCSPQuery.AFFILIATIONCHANGED, value: "Affiliation Changed", selected: ( OCSPQuery.AFFILIATIONCHANGED == this.model.revocationReason) } );
			reasons.push( { id: OCSPQuery.SUPERSEDED, value: "New Certificate Issued", selected: ( OCSPQuery.SUPERSEDED == this.model.revocationReason) } );
			reasons.push( { id: OCSPQuery.CESSATIONOFOPERATION, value: "CA Discontinued Operation", selected: ( OCSPQuery.CESSATIONOFOPERATION == this.model.revocationReason) } );

			if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_SUSPENDED
				&& this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED) {
				reasons.push( { id: OCSPQuery.CERTIFICATEHOLD, value: "Suspend Certificate", selected: true } );
			}

			if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUSPENDED) {
				reasons.push( { id: OCSPQuery.GOOD, value: "Resume Certificate", selected: true } );
			}

			reasons.push( { id: OCSPQuery.PRIVILEGEWITHDRAWN, value: "Privileges Withdrawn", selected: ( OCSPQuery.PRIVILEGEWITHDRAWN == this.model.revocationReason) } );

			var canRevoke = this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED;
			this.form.push(
			{
				id: "approval",
				legend: "msg.cr.revocation",
				fields: [
					{ id: "revocationReason", label: "msg.cr.revocationReason", type: "select", editable: canRevoke, value: reasons }
				]
			});
		}
	}

	return this.form;
}



CertificateRequestModel.prototype.isRegistrationOfficer = function(user) {
	return this.getRecipient().isRegistrationOfficer(user);
}



CertificateRequestModel.prototype.isAssignedToRegistrationOfficer = function(user) {
	if (!this.bo.assignedToRole) {
		return false;
	}

	var recipient = this.getRecipient();
	var roleId = recipient.getRARoleId();

	return recipient.isRegistrationOfficer(user) && (this.bo.assignedToRole == roleId);
}



CertificateRequestModel.prototype.isCertificationOfficer = function(user) {
	return this.getRecipient().isCertificationOfficer(user);
}



CertificateRequestModel.prototype.isManagerOfRequester = function(user) {
	return this.getOriginator().isManager(user);
}



CertificateRequestModel.prototype.isCATokenInHSMService = function() {
	var trustCenter = this.getRecipient();
	var ca = trustCenter.getX509CertificateIssuer(this.model.issuer);

	return ca.isOperational();
}



CertificateRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	var raOfficer = this.isAssignedToRegistrationOfficer(user);

	if (user.id == this.getOriginatorId() || raOfficer) {
		if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT)) {
			this.actionList.push("action.save");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			this.actionList.push("action.submit");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUBMIT) {
			this.actionList.push("action.cr.reqgen.usertoken");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_DELIVER) {
			this.actionList.push("action.cr.publish.usertoken");
		}
	}

	if (this.bo.lifecycle > ServiceRequestModel.LIFECYCLE_DELIVER) {
		this.appendRevocationActions(user, this.actionList);
	}

	if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) &&
		this.isRegistrationOfficer(user)) {
		this.actionList.push("action.save");
		this.actionList.push("action.approve");
		this.actionList.push("action.reject");
	}
	if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_PRODUCE) &&
		this.isCertificationOfficer(user)) {
		if (this.isCATokenInHSMService()) {
			this.actionList.push("action.cr.produce.hsmservice");
		}
	}

	return this.actionList;
}



CertificateRequestModel.prototype.appendRevocationActions = function(user, actionList) {
	if (this.isRegistrationOfficer(user) || this.isManagerOfRequester(user)) {
		if (this.hasCertificates()) {
			if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUSPENDED) {
				actionList.push("action.cr.updateStatus");
			} else if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED) {
				actionList.push("action.cr.revoke");
			}
		}
	}
}



/**
 * Can be changed in derived class to add more fields
 */
CertificateRequestModel.prototype.prepareServiceRequestMap = function() {
	var sr = {
		commonName: this.model.commonName
	}
	return sr;
}



CertificateRequestModel.prototype.transformSubjectDN = function() {
	var subject = this.getOriginator();

	var trustCenter = this.getRecipient();
	var ca = trustCenter.getX509CertificateIssuer(this.model.issuer);
	var icert = ca.getSignerCertificate();

	var dnmask = ca.getDNMask();
	var encoder = new DNEncoder(dnmask);

	var sr = this.prepareServiceRequestMap();
	encoder.setMap("servicerequest", sr);
	encoder.setDN("issuer", icert.getSubjectDNString());

	encoder.setMap("subject", subject.getSubjectMap());
	return encoder.encode();
}



CertificateRequestModel.prototype.validateSubjectDN = function(fld, val) {
	try	{
		var subjectDN = PKIXCommon.parseDN(val);
	}
	catch(e) {
		fld.message = e.message;
		return false;
	}
	this.model.subjectDN = subjectDN;
	return true;
}



CertificateRequestModel.prototype.getSubjectDN = function() {
	return this.model.subjectDN;
}



CertificateRequestModel.prototype.toApproval = function() {
	var raRoleId = this.getRecipient().getRARoleId();
	this.assignToRole(raRoleId);
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_APPROVE);
	this.setStatusInfo("Request submitted to RA for approval");
}



CertificateRequestModel.prototype.toProduction = function() {
	var caRoleId = this.getRecipient().getCARoleId();
	this.assignToRole(caRoleId);
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_PRODUCE);
	this.setStatusInfo("Request submitted to CA for production");
}



CertificateRequestModel.prototype.checkRevocationReason = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "checkRevocationReason(" + fld.id + "," + val + ")");

	if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUSPENDED
		&& val == OCSPQuery.CERTIFICATEHOLD) {
		GPSystem.log(GPSystem.DEBUG, module.id, "Certificate is already suspended");
		fld.message = "Certificate is already suspended";
		return false;
	}
	GPSystem.log(GPSystem.DEBUG, module.id, "passed...");

	return true;
}



CertificateRequestModel.prototype.updateModel = function(fld, val) {
	switch(fld.id) {
		case "revocationReason":
			if (!this.checkRevocationReason(fld, val)) {
				return false;
			}
			break;
		case "dn":
			return this.validateSubjectDN(fld, val);
	}
	this.model[fld.id] = val;
	return true;
}



CertificateRequestModel.prototype.handleAction = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleAction(" + user.id + "," + action + ")");

	switch(action) {
		case "action.save":
			this.handleSave();
			break;
		case "action.submit":
			this.updateDetails(this.model.commonName);
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_SUBMIT);
			this.setStatusInfo("Request being prepared for submission");
			break;
		case "action.approve":
			var caRoleId = this.getRecipient().getCARoleId();
			this.assignToRole(caRoleId);

			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_PRODUCE);
			this.setStatusInfo("Request approved");
			break;
		case "action.reject":
			this.assignToRole(0);
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_REJECTED);
			this.setStatusInfo("Request rejected");
			break;
		case "action.forward":
			this.toApproval();
			break;
		case "action.cr.produce.hsmservice":
			this.issueCertificate(this.service.hsmService, user);
			break;
		case "action.cr.revoke":
		case "action.cr.updateStatus":
			this.revokeCertificate();
			break;
	}
}



CertificateRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	this.handleAction(user, action);

	this.commit(user.id);
	return true;
}



CertificateRequestModel.prototype.handleSave = function() {
	if (this.getLifeCycle() == ServiceRequestModel.LIFECYCLE_NEW) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
		this.setStatusInfo("Request data entered");
	}
}



CertificateRequestModel.prototype.hasRequest = function() {
	return typeof(this.model.creq) != "undefined";
}



CertificateRequestModel.prototype.setRequest = function(request) {
	assert(request instanceof CVC, "Argument request must be of type CVC");
	this.model.creq = request.getBytes();
}



CertificateRequestModel.prototype.getRequest = function() {
	return new CVC(this.model.creq);
}



CertificateRequestModel.prototype.hasCertificates = function() {
	if (typeof(this.model.certIdList) == "undefined") {
		return 0;
	}
	return this.model.certIdList.length;
}



CertificateRequestModel.prototype.getCertificateIds = function() {
	return this.model.certIdList;
}



CertificateRequestModel.prototype.getCertificateHolder = function() {
	var cao = this.getRecipient();
	var holderDAO = this.service.daof.getHolderDAO();
	var certholder = holderDAO.getHolderBySubjectAndParent(this.getOriginatorId(), this.model.issuer, Holder.X509);
	if (!certholder) {
		var t = {
			name: this.model.commonName,
			subjectId: this.getOriginatorId()
		}
		var certholder = holderDAO.newHolderForParent(this.model.issuer, Holder.X509, t);
/*
		holderDAO.updateSubjectId(certholder, this.getOriginatorId());
		var name = this.model.commonName;
		var h = holderDAO.getHolderByTypeAndName(this.model.issuer, name, Holder.X509);
		if (h) {
			name += "(" + this.bo.id + ")";
		}
		holderDAO.updateName(certholder.id, this.model.commonName);
*/
	}

	return certholder;
}



CertificateRequestModel.prototype.generateKeyPair = function(cpf, deviceCert, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "generateKeyPair()");

	var trustCenter = this.getRecipient();

	// Determine or create holderId under the CA holder
	var certholder = this.getCertificateHolder();

	var ca = trustCenter.getX509CertificateIssuer(this.model.issuer);
	var policy = ca.getSubjectPolicyForRequest();

	policy.distinguishedName = this.model.subjectDN;
	policy.requestFormat = "sc-hsm";
	policy.overwriteKey = true;

	var signer = new X509Signer(this.service.daof, cpf, certholder);
	signer.setPolicy(policy);

	var signerName = trustCenter.getName() + "/" + this.model.commonName + " [" + Date() + "]";
	if (signerName.length > 100) {
		signerName = signerName.right(100);
	}

	var defaultId = KeyDomain.getDefaultKeyDomainFromCert(deviceCert);
	var template = {
		keyDomain: defaultId
	}

	var keyid = signer.newSigner(signerName, template);
	var req = signer.getRequestBinary(keyid);

	this.setRequest(new CVC(req));

	this.toApproval();
}



CertificateRequestModel.prototype.determineExtensions = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "determineExtensions()");

	// ToDo: Define extensions based on selected profile

	var extensions = [];
	var extvalues = this.getOriginator().getSubjectMap();

	var ext = new ASN1(ASN1.BIT_STRING, PKIXCommon.bitstringForInteger(PKIXCommon.digitalSignature | PKIXCommon.keyEncipherment));
	extensions.push( { oid: "id-ce-keyUsage", critical: true, value: ext.getBytes() } );

	var ext = new ASN1("extKeyUsages", ASN1.SEQUENCE);
	ext.add( new ASN1("keyPurposeId", ASN1.OBJECT_IDENTIFIER, new ByteString("id-kp-clientAuth", OID) ));
	ext.add( new ASN1("keyPurposeId", ASN1.OBJECT_IDENTIFIER, new ByteString("id-kp-emailProtection", OID) ));

	extensions.push( { oid: "id-ce-extKeyUsage", critical: false, value: ext.getBytes() } );

	if (extvalues.email) {
		var ext = new ASN1("subjectAltName", ASN1.SEQUENCE,
				   new ASN1("rfc822Name", 0x81, new ByteString(extvalues["email"], ASCII))
		);
		extensions.push( { oid: "id-ce-subjectAltName", critical: false, value: ext.getBytes() } );
	}

	return extensions;
}



CertificateRequestModel.prototype.issueCertificate = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "issueCertificate() with model.subjectDN " + this.model.subjectDN);

	var req = this.getRequest();

	var trustCenter = this.getRecipient();
	var ca = trustCenter.getX509CertificateIssuer(this.model.issuer);

	var certholder = this.getCertificateHolder();

	var cert = ca.issueCertificate(certholder, req.getPublicKey(), this.model.subjectDN, this.determineExtensions(), this.getId());
	var certId = cert.id;
	this.model.certIdList = [ certId ];

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_DELIVER);
	this.setStatusInfo("Certificate issued");
	if (this.model.role != undefined) {
		this.assignToRole(this.model.role);
	} else {
		this.assignToRole(0);
	}
}



CertificateRequestModel.prototype.activateCertificate = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "activateCertificate()");

	var certDAO = this.service.daof.getCertificateDAO();
	var cert = certDAO.getCertificateById(this.model.certIdList[0]);

	var certHolder = new Holder();
	certHolder.id = cert.holderId;

	var holderDAO = this.service.daof.getHolderDAO();
	holderDAO.updateCurrentCertificate(certHolder, cert);
}



CertificateRequestModel.prototype.deliverCertificate = function(sc, deviceCert, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "deliverCertificate()");

	var ks = new HSMKeyStore(sc);
	var certDAO = this.service.daof.getCertificateDAO();
	var cert = certDAO.getCertificateById(this.model.certIdList[0]);

	if (!cert) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "No certificate with ID " + this.model.certIdList[0] + " found");
	}

	var signerDAO = this.service.daof.getSignerDAO();
	var certHolder = new Holder();
	certHolder.id = cert.holderId;
	var signer = signerDAO.getSignerByKeyId(certHolder, cert.keyId);

	var defaultId = KeyDomain.getDefaultKeyDomainFromCert(deviceCert);

	if (!signer.keyDomain.equals(defaultId)) {
		var expected = signer.keyDomain.bytes(0, 8).toString(HEX) + "...";
		var given = defaultId.bytes(0, 8).toString(HEX) + "...";
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Wrong token: expected token with key domain " + expected + " but given " + given);
	}

	ks.storeEndEntityCertificate(cert.keyId, cert.bytes);

	this.activateCertificate();
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
	this.setStatusInfo("Certificate published");
}



CertificateRequestModel.prototype.revokeCertificate = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "revokeCertificate( reason " + this.model.revocationReason + " )");
	if (this.model.revocationReason == OCSPQuery.CERTIFICATEHOLD) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_SUSPENDED);
		this.setStatusInfo("Certificate on hold");
	} else if (this.model.revocationReason == OCSPQuery.GOOD) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
		this.setStatusInfo("Certificate resumed");
	} else {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_TERMINATED);
		this.setStatusInfo("Certificate revoked");
	}

	var invalidityDate = new Date();
	var certDAO = this.service.daof.getCertificateDAO();
	for each (var id in this.model.certIdList) {
		certDAO.updateRevocationStatus(id, this.model.revocationReason);
		if (this.model.revocationReason == OCSPQuery.GOOD) {
			certDAO.updateInvalidityDate(id, null);
		} else {
			certDAO.updateInvalidityDate(id, invalidityDate);
		}
		certDAO.updateRevocationDate(id, null);
	}
}



CertificateRequestModel.prototype.handleSmartCardHSM = function(sc, chain, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleCardAction for status: " + this.bo.lifecycle);
	switch(this.bo.lifecycle) {
		case ServiceRequestModel.LIFECYCLE_SUBMIT:
			var provider = new CryptoProvider(sc, chain.path);
			var cpf = { getCryptoProvider: function() { return provider }};
			this.generateKeyPair(cpf, chain.devicecert, user);
			break;
		case ServiceRequestModel.LIFECYCLE_PRODUCE:
			var ca = this.getRecipient();
			var keyDomain = KeyDomain.determineKeyDomain(sc, chain.devicecert);
			if (!ca.getKeyDomain().equals(keyDomain)) {
				GPSystem.log(GPSystem.DEBUG, module.id, "ca.keyDomain: " + ca.getKeyDomain());
				GPSystem.log(GPSystem.DEBUG, module.id, "current token keyDomain: " + keyDomain);
				this.setStatusInfo("The current token cannot issue certificates");
				break;
			}
			var provider = new CryptoProvider(sc, chain.path);
			var cpf = { getCryptoProvider: function() { return provider }};
			this.issueCertificate(cpf, user);
			break;
		case ServiceRequestModel.LIFECYCLE_DELIVER:
			this.deliverCertificate(sc, chain.devicecert, user);
			break;
		default:
			GPSystem.log(GPSystem.ERROR, module.id, "Unexpected handleCard in state " + this.bo.lifecycle);
			return;
	}

	if (typeof(user) == "undefined") {
		var id = this.getRecipientId();
	} else {
		var id = user.id;
	}
	this.commit(id);
}



CertificateRequestModel.prototype.handleCard = function(card, session, reqinfo) {
	var user = session.user;

	var sc = new SmartCardHSM(card);
	var devAutCert = sc.readBinary(SmartCardHSM.C_DevAut);
	var crypto = new Crypto();
	var chain = SmartCardHSM.validateCertificateChain(crypto, devAutCert);
	if (chain == null) {
		this.setStatusInfo("Could not authenticate SmartCard-HSM");
		return;
	}

	sc.openSecureChannel(crypto, chain.publicKey);

	this.handleSmartCardHSM(sc, chain, user);
}
