/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2018 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service Request model for PKCS#10 certificate requests
 */



var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var CertificateRequestModel	= require('pki-as-a-service/processes/CertificateRequestModel').CertificateRequestModel;
var PKCS10			= require('scsh/pkcs/PKCS10').PKCS10;
var X509Name			= require('scsh/x509/X509Name').X509Name;
var PKIXCommon			= require('scsh/x509/PKIXCommon').PKIXCommon;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function ESTRequestModel(service, bo) {
	CertificateRequestModel.call(this, service, bo);
	this.validationResult = [];
}

ESTRequestModel.prototype = Object.create(CertificateRequestModel.prototype);
ESTRequestModel.constructor = ESTRequestModel;

exports.ESTRequestModel = ESTRequestModel;


ESTRequestModel.prototype.getSubjectList = function(cn) {
	var trustCenter = this.getRecipient();
	var role = trustCenter.getRARoleId();
	var subjectDAO = this.service.daof.getSubjectDAO();
	var sublist = subjectDAO.getManagedSubjects(role);

	var sellist = [];

	for (var i = 0; i < sublist.length; i++) {
		var subject = sublist[i];
		var e = { id: subject.id, value: subject.name + " (" + subject.type + ")" };
		if (this.model.subjectId) {
			e.selected = this.model.subjectId == subject.id;
		} else {
			e.selected = cn == subject.name;
		}
		sellist.push(e);
	}

	return sellist;
}


ESTRequestModel.prototype.getForm = function(user) {
	if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_APPROVE) {
		if (this.form == undefined) {
			var editable = (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) && this.isRegistrationOfficer(user);

			var req = this.getRequest();

			this.form = [{
				id: "applicant",
				legend: "msg.cr.applicant",
				fields: [
					{ id: "csr", label: "msg.cr.csr", type: "text", size: 100, required: false, editable: false, value: req.toString() }
				]
				}
			];

			var subject = req.getSubject();
			var cn = "";
			if (subject.elements > 0) {
				var xName = new X509Name(subject)
				var reqDN = xName.toString();

				try {
					cn = xName.getRDNAsString(X509Name.commonName);
				} catch(e) {
					GPSystem.log(GPSystem.DEBUG, module.id, e);
				}
			}

			var subjectList = this.getSubjectList(cn);
			this.form[0].fields.push({ id: "subjectId", label: "msg.subject", type: "select", editable: editable, value: subjectList });

			if (this.model.subjectDN) {
				var dnval = PKIXCommon.dnToString(this.model.subjectDN);
			} else {
				var dnval = PKIXCommon.dnToString(this.transformSubjectDN());
			}

			this.form[0].fields.push(
				{ id: "dn", label: "msg.cr.dn", type: "text", size: 100, required: false, editable: editable, value: dnval }
			);

			this.form.push(
			{
				id: "approval",
				legend: "msg.cr.approval",
				fields: [
					{ id: "reviewRemark", label: "msg.cr.reviewRemark", type: "textarea", size: 4096, rows:10, cols:80, editable: editable, value: (this.model.reviewRemark ? this.model.reviewRemark : "") }
				]
			});

		}
	} else {
		this.form = CertificateRequestModel.prototype.getForm.call(this, user)
	}
	return this.form;
}



ESTRequestModel.prototype.checkPKCS10Request = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "checkPKCS10Request()");

	if (this.model.creq && val.length == 0) {
		// Allow empty submission
		// if the request was already uploaded
		return true;
	}

	try {
		if (val.byteAt(0) != 0x30) {
			var pem = val.toString(ASCII);
			GPSystem.log(GPSystem.DEBUG, module.id, "Converting PEM " + pem);
			var val = PKIXCommon.parsePEM("CERTIFICATE REQUEST", pem);
		}
		var req = new PKCS10(val);
		req.verify()
		this.model[fld.id] = val;
	}
	catch(e) {
		GPSystem.log(GPSystem.ERROR, module.id, e.message);
		fld.message = e.message;
		return false;
	}
	return true;
}



ESTRequestModel.prototype.updateModel = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "updateModel(" + fld.id + "," + val + ")");

	switch(fld.id) {
		case "creq":
			return this.checkPKCS10Request(fld, val);
	}
	return CertificateRequestModel.prototype.updateModel.call(this, fld, val);
}




ESTRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];


	var raOfficer = this.isAssignedToRegistrationOfficer(user);

	if ((user.id == this.getOriginatorId()) || raOfficer) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) {
			this.actionList.push("action.pkcs10.upload");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			this.actionList.push("action.save");
			this.actionList.push("action.submit");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_DELIVER) {
			this.actionList.push("action.pkcs10.accept");
			this.appendRevocationActions(user, this.actionList);
		}
	}

	if (this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_SUBMIT
		&& this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_DELIVER
	) {
		this.actionList = undefined;
		this.actionList = CertificateRequestModel.prototype.getActionList.call(this, user);
		GPSystem.log(GPSystem.DEBUG, module.id, "actionlist=" + this.actionList);
	}

	return this.actionList;
}



ESTRequestModel.prototype.updateOriginator = function() {
	var subject = this.service.getSubject(this.model.subjectId);
	var srDAO = this.service.daof.getServiceRequestDAO();
	srDAO.updateOriginatorId(this.bo, subject.bo);
	delete this.originator;
}



ESTRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.pkcs10.upload":
			this.handleUpload();
			break;
		case "action.submit":
			this.submitRequest();
			break;
		case "action.pkcs10.accept":
			this.activateCertificate();
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
			this.setStatusInfo("Certificate accepted");
			break;
		case "action.approve":
			this.updateOriginator();
			return CertificateRequestModel.prototype.perform.call(this, user, action);
		default:
			return CertificateRequestModel.prototype.perform.call(this, user, action);
	}

	this.commit(user.id);
	return true;
}



/**
 * Perform periodic or ad-hoc activities
 */
ESTRequestModel.prototype.ping = function(req, res) {

	if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_PRODUCE) {
		if (this.isCATokenInHSMService()) {

			if (this.model.subjectDN == undefined) {
				this.model.subjectDN = this.transformSubjectDN();
			}

			this.model.reviewRemark = "Automatically approved";
			this.issueCertificate();
			this.commit(0);
		} else {
			GPSystem.log(GPSystem.INFO, module.id, "CA token not online or available");
			res.setStatus(HttpResponse.SC_ACCEPTED);
		}
	} else if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_DELIVER) {
		this.activateCertificate();
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
		this.setStatusInfo("Certificate published");
		this.commit(0);
	}
}



ESTRequestModel.prototype.getRequest = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "getRequest()");

	if (this.model.creq.byteAt(0) != 0x30) {
		var pem = this.model.creq.toString(ASCII);
		GPSystem.log(GPSystem.DEBUG, module.id, "Converting PEM " + pem);
		var req = PKIXCommon.parsePEM("CERTIFICATE REQUEST", pem);
		return new PKCS10(req);
	}

	GPSystem.log(GPSystem.DEBUG, module.id, this.model.creq);
	return new PKCS10(this.model.creq);
}



ESTRequestModel.prototype.handleUpload = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleUpload()");
	var req = this.getRequest();
	var subject = req.getSubject();
	var xName = new X509Name(subject)
	try {
		this.model.commonName = xName.getRDNAsString(X509Name.commonName);
	} catch(e) {
		GPSystem.log(GPSystem.DEBUG, module.id, e);
	}
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
}



ESTRequestModel.prototype.submitRequest = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "submitRequest()");

	this.getCertificateHolder();

	this.toApproval();
}
