/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Card related actions performed in the UI
 */



CardAction			= require('scsh/srv-cc1/CardAction').CardAction;

HSMKeyStore			= require('scsh/sc-hsm/HSMKeyStore').HSMKeyStore;


function InitializationWizard(ui, sr, redirectUrl) {
	CardAction.call(this, ui);
	this.sr = sr;
	this.redirectUrl = redirectUrl;
	this.requestPIN = -1;
	GPSystem.log(GPSystem.DEBUG, module.id, "new InitializationWizard(" + sr + "," + redirectUrl + ")");

	// default values
	this.noOfKeyDomains = 4;
	this.state = "derivationKey";

	this.initWizardFromSR();
}


InitializationWizard.prototype = Object.create(CardAction.prototype);
InitializationWizard.constructor = InitializationWizard;

exports.InitializationWizard = InitializationWizard;


InitializationWizard.USER_PIN = "User PIN";
InitializationWizard.TRANSPORT_PIN = "Transport PIN";
InitializationWizard.PUBLIC_KEY_AUTHENTICATION = "Public Key Authentication";

InitializationWizard.WITH_RRC = 4;
InitializationWizard.WITHOUT_RRC = 5;

InitializationWizard.SCHEME_KEY_DOMAINS = 6;
InitializationWizard.SCHEME_NO_DKEK = 7;
InitializationWizard.SCHEME_RANDOM_DKEK = 8;
InitializationWizard.SCHEME_DKEK_SHARES = 9;

InitializationWizard.prototype.handleRequest = function(req, res) {
	default xml namespace = "http://www.w3.org/1999/xhtml";
	GPSystem.log(GPSystem.DEBUG, module.id, "handleRequest");

	if (req.method == "POST" && this.state != CardAction.SUCCESS) {
		if (req.parameters.change) {
			this.ui.wizard = new InitializationWizard(this.ui, this.sr, this.redirectUrl);
			this.ui.redirectTo(req, res, req.url, "/wizard");
		} else if (req.parameters.keyDomain) {
			this.sr.generateDerivationKey(req.parameters.keyDomain, req.parameters.derivationKey);

			this.keyDomain = req.parameters.keyDomain;
			this.derivationKey = req.parameters.derivationKey;
			this.state = "rrcMode";
		} else if (req.parameters.derivationKey) {
			this.derivationKey = req.parameters.derivationKey;
			this.state = "rrcMode";
		} else if (req.parameters.rrcMode) {
			this.rrcMode = req.parameters.rrcMode;
			this.state = "authMech";
		} else if (req.parameters.authMech) {
			this.authMech = req.parameters.authMech;
			this.state = "pin";
		} else if (req.parameters.pin) {
			this.pin = req.parameters.pin;
			this.state = "noOfKeyDomains";
		} else if (req.parameters.noOfKeyDomains) {
			this.noOfKeyDomains = parseInt(req.parameters.noOfKeyDomains);
			this.state = "summary";
		} else if (req.parameters.initialize) {

			this.sr.setInitializeData(
				this.ui.session.user,
				{
					derivationKeyLabel: this.derivationKey,
					rrcMode: this.rrcMode == InitializationWizard.WITH_RRC,
					transportMode: this.authMech == InitializationWizard.TRANSPORT_PIN,
					pin: this.pin,
					noOfKeyDomains: this.noOfKeyDomains
				}
			);

			this.state = CardAction.NEW;
		} else if (req.parameters.cancel) {
			GPSystem.log(GPSystem.DEBUG, module.id, "redirect to " + this.redirectUrl);
			this.ui.redirectTo(req, res, req.url, this.redirectUrl);
			return;
		}

		this.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	var page = this.getPage(req, res, req.url);
	if (page) {
		var t = this.ui.generateTemplate(req.url);
		var c = t..div.(@id == "content");
		c.div = page;
		this.ui.sendPage(req, res, req.url, t);
	} else {
		GPSystem.log(GPSystem.DEBUG, module.id, "redirect to " + this.redirectUrl);
		this.ui.redirectTo(req, res, req.url, this.redirectUrl);
	}
}



InitializationWizard.prototype.initWizardFromSR = function() {
	if (this.sr.getDerivationKeyLabel()) {
		this.derivationKey = this.sr.getDerivationKeyLabel();

		if (this.sr.getDerivationKey()) {
			this.state = "rrcMode";
		} else {
			this.state = "connectKey";
		}
	}
}



InitializationWizard.prototype.getDerivationKeyPage = function(url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var hsmList = this.sr.getHSMList();

	if (hsmList.length == 0) {
		var page =
			<div>
				<div id="error"/>

				<h1>Initialize Token Wizard</h1>

				<h3>SO-PIN</h3>

				<p>The SO-PIN will be derived from a derivation key of one the connected trust center token.</p>

				<p>There is currently no trust center token online.</p>

				<p>Please connect a trust center token to select a derivation key. <a href="">continue</a>.</p>

				<pre> java -jar ocf-client.jar -r {"\"<reader name>\" " + ApplicationServer.instance.serverURL + "/rt/hsm"}</pre>
			</div>
	} else {
		var page =
			<div>
				<div id="error"/>

				<h1>Initialize Token Wizard</h1>

				<h3>SO-PIN</h3>

				<p>The SO-PIN will be derived from a derivation key of one the connected trust center token.</p>

				<form class="pure-form pure-form-aligned" action="" method="post" enctype="multipart/form-data">
					<fieldset>
						<legend>Please select a derivation key ...</legend>
						<div class="pure-control-group">
							<label>Key</label>
							<select name="derivationKey"/>
							<button class="pure-button" type="submit">Select</button>
						</div>
					</fieldset>
				</form>

				<form class="pure-form pure-form-aligned" action="" method="post" enctype="multipart/form-data">
					<fieldset>
						<legend>... or generate a new derivation key</legend>
						<div class="pure-control-group">
							<label>Key Domain</label>
							<select name="keyDomain"/>
						</div>
						<div class="pure-control-group">
							<label>Label</label>
							<input id="derivationKey" name="derivationKey" type="text" value={ "SO-PIN-SR" + this.sr.getId() } />
							<button class="pure-button" type="submit">Generate</button>
						</div>
					</fieldset>
				</form>
			</div>

		var keyList = page.form[0].fieldset.div.select;
		var keyDomainList = page.form[1].fieldset.div[0].select;

		for (var i = 0; i < hsmList.length; i++) {
			var hsm = hsmList[i];

			// Populate key list

			var keys = this.getSymmetricKeys(hsm.sc);

			var optgroup = <optgroup label={ hsm.path } />;

			for (var j = 0; j < keys.length; j++) {
				var key = keys[j];

				var option = <option value={key.getLabel()}>{key.getLabel()}</option>

				if (key.getLabel() == this.derivationKey) {
					option.@selected = "selected";
				}

				optgroup.appendChild(option);
			}

			if (keys.length > 0) {
				keyList.appendChild(optgroup);
			}

			// Populate key domain list

			var group = <optgroup label={ hsm.path } />;

			for (var j = 0; j < hsm.keyDomain.length; j++) {
				var kdid = hsm.keyDomain[j].toString(HEX);
				var shortenedKDID = kdid;
				if (shortenedKDID.length > 16) {
					shortenedKDID = shortenedKDID.substring(0, 16) + "...";
				}

				var o = <option value={kdid}>{shortenedKDID}</option>

				group.appendChild(o);
			}

			if (hsm.keyDomain.length > 0) {
				keyDomainList.appendChild(group);
			}

		}

		if (keyList.*.length() == 0) {
			keyList.appendChild( <option value="n/a" disabled="disabled=" selected = "selected">No symmetric key found</option> );
			page.form[0].fieldset.div.button.@disabled = "disabled";
		}
	}

	return page;
}



InitializationWizard.prototype.getConnectKeyPage = function(url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page =
		<div>
			<div id="error"/>

			<h1>Initialize Token Wizard</h1>

			<h3>SO-PIN</h3>

			<p>The SO-PIN will be derived from the derivation key with label <strong>{this.derivationKey}</strong>.</p>

			<p>There is currently no trust center token online which contains this derivation key.</p>

			<p>Please connect the corresponding trust center token to <a href="">continue</a>.</p>

			<pre> java -jar ocf-client.jar -r {"\"<reader name>\" " + ApplicationServer.instance.serverURL + "/rt/hsm"}</pre>
		</div>

	return page;
}



InitializationWizard.prototype.getSymmetricKeys = function(sc) {
	var aesKeys = [];

	var keyStore = new HSMKeyStore(sc);
	var labels = keyStore.enumerateKeys();
	for (var i = 0; i < labels.length; i++) {
		var label = labels[i];
		GPSystem.log(GPSystem.DEBUG, module.id, "label: " + label);
		var key = keyStore.getKey(label);
		GPSystem.log(GPSystem.DEBUG, module.id, "type: " + key.getType());
		if (key.getType() == "AES") {
			GPSystem.log(GPSystem.DEBUG, module.id, "Add AES Key: " + label);
			aesKeys.push(key);
		}
	}

	return aesKeys;
}



InitializationWizard.prototype.getConfigPage = function(req, res, url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = <div>
		<div id="error"/>

		<h1>Initialize Token Wizard</h1>

		<h3>Configure Initialization Data</h3>

		<p>Please configure the initialization options.</p>

		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<label>Authentication Mechanism</label>
			<select name="authMech">
				<option value={InitializationWizard.TRANSPORT_PIN}>Transport PIN</option>
				<option value={InitializationWizard.USER_PIN}>User PIN</option>
			</select>
			<button class="pure-button" type="submit">Submit</button>
		</form>
	</div>

	return page;
}



InitializationWizard.prototype.getAllowRRCPage = function(req, res, url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = <div>
		<div id="error"/>

		<h1>Initialize Token Wizard</h1>

		<h3>Configure Initialization Data</h3>

		<p>Please configure the initialization options.</p>

		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<label>Resetting PIN with SO-PIN</label>
			<select name="rrcMode">
				<option value={InitializationWizard.WITH_RRC}>Allowed</option>
				<option value={InitializationWizard.WITHOUT_RRC}>Not Allowed</option>
			</select>
			<button class="pure-button" type="submit">Submit</button>
		</form>
	</div>

	return page;
}



InitializationWizard.prototype.getEnterPINPage = function(req, res, url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	if (this.authMech == InitializationWizard.TRANSPORT_PIN) {
		var pinLabel = "Transport PIN";
	} else {
		var pinLabel = "PIN";
	}

	var crypto = new Crypto();
	this.pin = this.sr.service.generateActivationCode(6);

	var page = <div>
		<div id="error"/>

		<h1>Initialize Token Wizard</h1>

		<h3>Configure Initialization Data</h3>

		<p>Please configure the initialization options.</p>

		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<label>{pinLabel}</label>
			<input id="pin" name="pin" type="text" value={ this.pin } />
			<button class="pure-button" type="submit">Submit</button>
		</form>
	</div>

	return page;
}



InitializationWizard.prototype.getDeviceKeyEncryptionSchemePage = function(req, res, url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = <div>
		<div id="error"/>

		<h1>Initialize Token Wizard</h1>

		<h3>Configure Initialization Data</h3>

		<p>Please configure the initialization options.</p>

		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<label>Device Key Encryption Scheme</label>
			<select name="scheme">
				<option value={InitializationWizard.SCHEME_KEY_DOMAINS}>Key Domains</option>
				<option value={InitializationWizard.SCHEME_NO_DKEK}>NO DKEK</option>
			</select>
			<button class="pure-button" type="submit">Submit</button>
		</form>
	</div>

	return page;
}



InitializationWizard.prototype.getNoOfKeyDomainsPage = function(req, res, url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = <div>
		<div id="error"/>

		<h1>Initialize Token Wizard</h1>

		<h3>Configure Initialization Data</h3>

		<p>Please configure the initialization options.</p>

		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<label>Number of Key Domains</label>
			<input id="noOfKeyDomains" name="noOfKeyDomains" type="number" max="255" value={ this.noOfKeyDomains } />
			<button class="pure-button" type="submit">Submit</button>
		</form>
	</div>

	return page;
}



InitializationWizard.prototype.getInitializePage = function(req, res, url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var page = <div>
		<div id="error"/>

		<h1>Initialize Token Wizard</h1>

		<h3>Summary</h3>

		<table class="pure-table pure-table-horizontal form-table">
				<tr>
					<th>Token</th>
					<td>{this.sr.getTokenPath()}</td>
				</tr>
				<tr>
					<th>SO-PIN Derivation Key</th>
					<td>{this.derivationKey}</td>
				</tr>
				<tr>
					<th>Reset Retry Counter</th>
					<td>{this.rrcMode == InitializationWizard.WITH_RRC ? "Allowed" : "Not Allowed"}</td>
				</tr>
				<tr>
					<th>Authentication Mechanism</th>
					<td>{this.authMech}</td>
				</tr>
				<tr>
					<th>PIN</th>
					<td>{this.pin}</td>
				</tr>
				<tr>
					<th>Number of Key Domains</th>
					<td>{this.noOfKeyDomains}</td>
				</tr>
		</table>

		<p>
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
				<button class="pure-button" type="submit" name="initialize" value="true">Initialize</button>
				<button class="pure-button" type="submit" name="change" value="true">Change</button>
				<button class="pure-button" type="submit" name="cancel" value="true">Cancel</button>
			</form>
		</p>
	</div>

	return page;
}



InitializationWizard.prototype.getPage = function(req, res, url) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	switch(this.state) {
		case "derivationKey":
			var page = this.getDerivationKeyPage(url);
			break;
		case "connectKey":
			if (!this.sr.getDerivationKey()) {
				var page = this.getConnectKeyPage();
				break;
			}
			this.state = "rrcMode";
		case "rrcMode":
			var page = this.getAllowRRCPage();
			break;
		case "authMech":
			var page = this.getConfigPage();
			break;
		case "pin":
			var page = this.getEnterPINPage();
			break;
		case "noOfKeyDomains":
			var page = this.getNoOfKeyDomainsPage();
			break;
		case "summary":
			var page = this.getInitializePage();
			break;
		case CardAction.NEW:
			this.state = CardAction.NOCLIENT;
			var page = this.getProgressPage(req, res);
			break;
		case CardAction.NOCLIENT:
			var page = this.getNoClientPage(req, res);
			this.state = CardAction.NEW;
			break;
		case CardAction.ERROR:
			var page = this.getErrorPage(req, res);
			this.state = CardAction.NEW;
			break;
		case CardAction.SUCCESS:	// reached after handleCard
			return null;
			break;
		default:
			var page = this.getDerivationKeyPage();
	}

	return page;
}





InitializationWizard.prototype.handleCardAction = function(card, session, pathInfo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleCardAction");

	this.state = CardAction.ERROR;

	var sc = new SmartCardHSM(card);
	var devAutCert = sc.readBinary(SmartCardHSM.C_DevAut);
	var crypto = new Crypto();
	var chain = SmartCardHSM.validateCertificateChain(crypto, devAutCert);
	if (chain == null) {
		this.reportError("Could not authenticate SmartCard-HSM");
		return;
	}

	if (this.sr.getTokenPath() != chain.path) {
		this.reportError("Wrong token presented. Expecting token " + this.sr.getTokenPath());
		return;
	}

	try {
		this.sr.initialize(card, chain, session.user);
		this.state = CardAction.SUCCESS;
	} catch (e) {
		this.reportError("Initialization failed: " + e.message);
		GPSystem.log(GPSystem.ERROR, module.id, "Initialization failed: " + e.message + "\n" + e.stack);
	}
}
