/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A service request model for registering a new token
 */

ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
Subject				= require('scsh/pki-db/Subject').Subject;
RoleDAO				= require('scsh/pki-db/db/RoleDAO').RoleDAO;



/**
 * Data model for registering  a token
 *
 *
 * @constructor
 */
function RegisterMyTokenRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
	this.validationResult = [];
}

RegisterMyTokenRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
RegisterMyTokenRequestModel.constructor = RegisterMyTokenRequestModel;

exports.RegisterMyTokenRequestModel = RegisterMyTokenRequestModel;



RegisterMyTokenRequestModel.STATE_WRONG_ACTIVATION_CODE = "Wrong Activation Code";
RegisterMyTokenRequestModel.STATE_COMPLETED = "Token Registered";

RegisterMyTokenRequestModel.INITIAL_ACTIVATION_TRIES = 3;
RegisterMyTokenRequestModel.NO_OF_INITIAL_ACTIVATION_REQUESTS = 3;


RegisterMyTokenRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		var editable = (user.id == this.getOriginatorId()) &&
			((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) || (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT));

		var changeableEmail = this.isAssignedToRole(user) && this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_TERMINATED;

		this.form = [{
			id: "regdata",
			legend: "msg.registration.data",
			fields: [
				{ id: "token", label: "msg.registration.token", type: "text", size: 50, required: true, editable: editable, value: (this.model.tokenId ? this.model.tokenId: "") },
				{ id: "email", label: "msg.registration.email", type: "text", size: 50, required: true, editable: changeableEmail, value: (this.model.email ? this.model.email: "") },
				{ id: "activationTries", label: "msg.registration.activationTries", type: "number", required: true, editable: editable, value: (this.model.activationTries ? this.model.activationTries: "") }
			]}
		];
	}

	return this.form;
}



RegisterMyTokenRequestModel.prototype.isAssignedToRole = function(user) {
	if (!this.bo.assignedToRole) {
		return false;
	}

	return (typeof(user.roles) != "undefined") && (user.roles.indexOf(this.bo.assignedToRole) >= 0);
}



RegisterMyTokenRequestModel.prototype.assignToRegistrationOfficer = function() {
	if (this.model.raRoleID) {
		this.assignToRole(this.model.raRoleID);
	} else {
		this.assignToRole(RoleDAO.ID_RegistrationManager);
	}
}



RegisterMyTokenRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	switch(this.bo.lifecycle) {
		case ServiceRequestModel.LIFECYCLE_TERMINATED:
			return this.actionList;
		case ServiceRequestModel.LIFECYCLE_REJECTED:
		case ServiceRequestModel.LIFECYCLE_COMPLETE:
			this.actionList.push("action.registration.terminate");
			return this.actionList;
	}

	if (this.isAssignedToRole(user)) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_ERROR) {
			if (this.model.activationRequests <= 0) {
				this.actionList.push("action.registration.unblock_activation_requests");
			} else if (this.model.activationTries <= 0) {
				this.actionList.push("action.registration.unblock_activation_code");
			}
			this.actionList.push("action.registration.reject");
		} else {
			this.actionList.push("action.registration.change_email");
			this.actionList.push("action.registration.terminate");
		}
	}

	return this.actionList;
}



/**
 * Helper function to check if the given email is valid
 *
 * @param {String} email the e-mail address
 */
RegisterMyTokenRequestModel.prototype.isValidEmail = function(email) {
	GPSystem.log(GPSystem.DEBUG, module.id, "isValidEmail( " + email + " )");

	if (email.length <= 0 ||
		(email.match(/[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+/) == null) ||
		(email.match(/[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+/)[0] != email)) {
			this.addMessage("Failed - Invalid e-mail address " + email);
			GPSystem.log(GPSystem.DEBUG, module.id, "Invalid e-mail address " + email);
			return false;
	}

	return true;
}



RegisterMyTokenRequestModel.prototype.checkEmail = function(fld, email) {
	GPSystem.log(GPSystem.DEBUG, module.id, "checkEmail( " + email + " )");

	if (this.model.email == email) {
		// Return true to allow modifications of the name
		return true;
	}

	if (email.length <= 0 ||
		(email.match(/[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+/) == null) ||
		(email.match(/[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+/)[0] != email)) {
			this.addMessage("Failed - Invalid e-mail address " + email);
			fld.message = "Invalid e-mail address " + email;
			GPSystem.log(GPSystem.DEBUG, module.id, "Invalid e-mail address " + email);
			return false;
	}

	var subjectDAO = this.service.daof.getSubjectDAO();
	var subject = subjectDAO.getSubjectByEmail(email);
	if (subject) {
		fld.message = "The email " + email + " was already assigned. Please choose a different email.";
		return false;
	}

	this.model.email = email;

	return true;
}



RegisterMyTokenRequestModel.prototype.validateField = function(fld, val) {
	switch(fld.id) {
		case "email":
			try {
				if (!this.checkEmail(fld, val)) {
					fld.value = val;
					return false;
				};
			}
			catch(e) {
				GPSystem.log(GPSystem.ERROR, module.id, e.message);
				fld.message = e.message;
				fld.value = val;
				return false;
			}
			break;
		default:
			this.validateFieldDefault(fld, val);
	}
	return true;
}



RegisterMyTokenRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.registration.change_email":
			this.model.activationRequests = RegisterMyTokenRequestModel.NO_OF_INITIAL_ACTIVATION_REQUESTS;
			this.model.activationTries = RegisterMyTokenRequestModel.INITIAL_ACTIVATION_TRIES;
			this.sendNewActivationCode();
			this.setStatusInfo("Changed e-mail address, reset activation code and request counter");
			break;
		case "action.registration.unblock_activation_code":
			this.model.activationTries = RegisterMyTokenRequestModel.INITIAL_ACTIVATION_TRIES;
			this.sendNewActivationCode();
			this.setStatusInfo("Activation Code unblocked");
			break;
		case "action.registration.unblock_activation_requests":
			this.model.activationRequests = RegisterMyTokenRequestModel.NO_OF_INITIAL_ACTIVATION_REQUESTS;
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_VALIDATE);
			this.setStatusInfo("Requesting new Activation Codes unblocked");
			break;
		case "action.registration.reject":
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_REJECTED);
			this.setStatusInfo("Service Request rejected");
			break;
		case "action.registration.terminate":
			this.deregisterToken();
			this.service.removeRegistrationTokenAction(this.model.tokenId);
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_TERMINATED);
			this.setStatusInfo("Token deregistered");
			break;
	}

	this.commit(user.id);
	return true;
}



RegisterMyTokenRequestModel.prototype.getToken = function() {
	var tokenDAO = this.service.daof.getTokenDAO();
	return tokenDAO.getToken(this.model.tokenId);
}



RegisterMyTokenRequestModel.prototype.deregisterToken = function() {
	var tokenDAO = this.service.daof.getTokenDAO();
	tokenDAO.deregisterToken(this.model.tokenId);
}



RegisterMyTokenRequestModel.prototype.setSubject = function(email) {
	var subjectDAO = this.service.daof.getSubjectDAO();
	var subject = subjectDAO.getSubjectByEmail(email);

	if (!subject) {
		var template = {
			type: "Person"
		};
		subject = subjectDAO.newSubject(template);

		if (subject.id == 1) {
			// Set special role "systemadmin" for first user in order to bootstrap the system
			var roleDAO = this.service.daof.getRoleDAO();
			roleDAO.createPredefinedRoles();
			roleDAO.newAssignedRole(subject.id, RoleDAO.ID_SystemAdmin, this.bo.id);
			roleDAO.newAssignedRole(subject.id, RoleDAO.ID_HSMAdmin, this.bo.id);
			roleDAO.newAssignedRole(subject.id, RoleDAO.ID_Auditor, this.bo.id);
			roleDAO.newAssignedRole(subject.id, RoleDAO.ID_SubscriberManager, this.bo.id);
			roleDAO.newAssignedRole(subject.id, RoleDAO.ID_Subscriber, this.bo.id);
			roleDAO.newAssignedRole(subject.id, RoleDAO.ID_RegistrationManager, this.bo.id);
		}
	}

	var srDAO = this.service.daof.getServiceRequestDAO();
	srDAO.updateOriginatorId(this.bo, subject);
}



/**
 * Generate random data and decimalize into a sequence of digits
 *
 * @param {digits} number of digits
 * @type String
 * @return the random activation code
 */
RegisterMyTokenRequestModel.prototype.generateActivationCode = function(digits) {
	var crypto = new Crypto();
	var raw = crypto.generateRandom(digits);

	var c;
	do	{
		var str = raw.toString(HEX);
		c = digits;
		var result = "";
		var i = 0;

		while ((i < str.length) && (c > 0)) {
			var d = str.charCodeAt(i);
			if ((d >= 0x30) && (d <= 0x39)) {
				result = result.concat(String.fromCharCode(d));
				c--;
			}
			i++;
		}
		if (c > 0) {
			raw = raw.not();
		}

	} while (c > 0);

	return result;
}



/**
 * Generate a new Activation Code and send it to
 * the given e-mail address.
 */
RegisterMyTokenRequestModel.prototype.sendNewActivationCode = function() {
	if (this.service.testMode) {
		/* use this activation code generator only for testing*/
		var crypto = new Crypto()
		var hash = crypto.digest(Crypto.SHA_256, new ByteString(this.model.email, ASCII));
		this.model.activationCode = hash.bytes(0,3).toString(HEX);
	} else {
		/* use this activation code generator for productive system */
		this.model.activationCode = this.generateActivationCode(6);
	}

	GPSystem.log(GPSystem.DEBUG, module.id, "ACTIVATION CODE " + this.model.activationCode);

	this.setStatusInfo("Activation code send");
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_VALIDATE);
	if (!this.service.testMode) {
		this.service.sendActivationCode(this);
	}
}



RegisterMyTokenRequestModel.prototype.getEmail = function() {
	return this.model.email;
}



RegisterMyTokenRequestModel.prototype.getActivationCode = function() {
	return this.model.activationCode
}



RegisterMyTokenRequestModel.prototype.getActivationTries = function() {
	return this.model.activationTries;
}



RegisterMyTokenRequestModel.prototype.isCompleted = function() {
	return this.getStatusInfo() == RegisterMyTokenRequestModel.STATE_COMPLETED;
}



/**
 * Assign a valid e-mail address to a subject and send an activation code.
 * Sets the service request life-cycle to LIFECYCLE_VALIDATE.
 *
 * @param {String} email the e-mail address
 */
RegisterMyTokenRequestModel.prototype.setEmail = function(email) {
	assert(email, "Parameter email must not be empty");
	assert(typeof(email) == "string", "Parameter email must be a string");

	if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_NEW &&
		this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_VALIDATE) {
		GPSystem.log(GPSystem.ERROR, module.id, "setEmail not allowed in life-cycle " + this.bo.lifecycle);
		return;
	}

	this.model.activationRequests--;
	if (this.model.activationRequests <= 0) {
		this.setStatusInfo("Activation Code Generation Limit Reached");
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_ERROR);
		this.assignToRegistrationOfficer();
		this.commit();
		return;
	}

	if (!this.model.email || !this.model.raRoleID) {
		this.model.email = email
		if (!this.isValidEmail(email)) {
			this.commit();
			return;
		}
	}

	this.sendNewActivationCode();
	this.commit();
}



/**
 * Verify the activation code send via mail to the user.
 * Each verification decreases the left activation tries.
 * A failed verification sets the service request status to
 * STATE_WRONG_ACTIVATION_CODE.
 * If there are no more activation tries the service request
 * will be assigned to the corresponding registration officer.
 * After a successful verifcation the status is set to STATE_COMPLETED.
 *
 * @param {String} code the activation code
 */
RegisterMyTokenRequestModel.prototype.verifyActivationCode = function(code) {
	if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_VALIDATE) {
		GPSystem.log(GPSystem.ERROR, module.id, "verifyActivationCode not allowed in life-cycle " + this.bo.lifecycle);
		return;
	}

	// Try activation code
	this.model.activationTries--;

	if (this.model.activationCode != code) {
		if (this.model.activationTries <= 0) {
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_ERROR);
			this.assignToRegistrationOfficer();
		}
		this.setStatusInfo(RegisterMyTokenRequestModel.STATE_WRONG_ACTIVATION_CODE);
		this.commit();
		return;
	}

	// e-mail address verified
	this.setSubject(this.model.email);
	var tokenDAO = this.service.daof.getTokenDAO();
	tokenDAO.updateSubjectId(this.getToken(), this.bo.originatorId);
	tokenDAO.updateServiceRequestId(this.getToken(), this.bo.id);
	var subjectDAO = this.service.daof.getSubjectDAO();
	var subject = new Subject(null, {id: this.bo.originatorId}); // wrap subject id
	subjectDAO.updateEmail(subject, this.model.email);

	this.setStatusInfo(RegisterMyTokenRequestModel.STATE_COMPLETED);
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
	this.service.removeRegistrationTokenAction(this.model.tokenId);
	this.commit();
}



/**
 * Cancel the RegisterMyTokenServiceRequest during enrollment.
 * Called by the user via RegistrationWizard.
 */
RegisterMyTokenRequestModel.prototype.cancelServiceRequest = function() {
	if (this.model.raRoleID) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_RETURNED);
		this.assignToRole(this.model.raRoleID);
		this.commit();
	}
}



/**
 * Handle Card Updater Calls
 */
RegisterMyTokenRequestModel.prototype.handleSmartCardHSM = function(sc, chain, user) {
	// Return true to keep the token action
	return true;
}
