/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A service request importing an existing X509 Signer from a Smartcard-HSM
 */

ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
HSMKeyStore			= require('scsh/sc-hsm/HSMKeyStore').HSMKeyStore;
PKIXCommon			= require('scsh/x509/PKIXCommon').PKIXCommon;
KeyDomain			= require('pki-as-a-service/service/KeyDomain').KeyDomain;
Holder				= require('scsh/pki-db/Holder').Holder;
X509Signer			= require('scsh/x509/X509Signer').X509Signer;



function ImportX509SignerRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
}

ImportX509SignerRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
ImportX509SignerRequestModel.constructor = ImportX509SignerRequestModel;

exports.ImportX509SignerRequestModel = ImportX509SignerRequestModel;



ImportX509SignerRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		this.form = [];

		var editable = false;
		if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT) {
			editable = true;
		}

		if (!this.model.rootKeySpecId) {
			this.model.rootKeySpecId = 1;
			this.model.subKeySpecId = 1;
			this.model.supportedProfiles = [];
		}

		var maxValidity = 18263;
		var defaultValidityDays = 3650;
		var maxPath = 9;
		var defaultPath = 0;

		var defaultSubCertValidityDays = 730;
		if (defaultSubCertValidityDays > maxValidity) {
			defaultSubCertValidityDays = maxValidity;
		}

		var subkeyspeclist = [];
		var keyspecs = this.service.getKeySpecificationList();
		for (var i = 0; i < keyspecs.length; i++) {
			var keyspec = keyspecs[i];

			var e = { id: keyspec.id, value: keyspec.name };
			if (this.model.subKeySpecId == keyspec.id) {
				e.selected = true;
			}
			subkeyspeclist.push(e);
		}

		var profiles = this.service.getCertificateProfileList();
		var profilelist = [];

		for (var i = 0; i < profiles.length; i++) {
			var profile = profiles[i];

			var e = { id: profile.id, value: profile.name };
			if (this.model.supportedProfiles.indexOf(profile.id) >= 0) {
				e.selected = true;
			}
			profilelist.push(e);
		}

		var subjectSpec = {
			id: "subjectspec",
			legend: "msg.impsigner.subjectspec",
			fields: [
				{ id: "subKeySpecId", label: "msg.impsigner.subKeySpecId", type: "select", editable: editable, value: subkeyspeclist },
				{ id: "subCertValidityDays", label: "msg.impsigner.subCertValidityDays", type: "number", min: 1, max: maxValidity, required: true, editable: editable, value: (this.model.subCertValidityDays ? this.model.subCertValidityDays : defaultSubCertValidityDays) },
				{ id: "dnmask", label: "msg.impsigner.dnmask", type: "text", size: 320, required: true, editable: editable, value: (this.model.dnmask ? this.model.dnmask : "c=${issuer.c},o=${issuer.o},cn=${servicerequest.commonName}") },
				{ id: "supportedProfiles", label: "msg.impsigner.supportedProfiles", type: "select", multiselect: true, editable: editable, value: profilelist }
			]
		}

		this.form.push(subjectSpec);

		var certSpec = {
			id: "rootcertspec",
			legend: "msg.xsigner.cacertspec",
			fields: [
				{ id: "caPathLen", label: "msg.impsigner.caPathLen", type: "number", min: 0, max: maxPath, required: true, editable: editable, value: (this.model.caPathLen ? this.model.caPathLen : defaultPath) }
			]
		}

		this.form.push(certSpec);

		var crl = {
			id: "crl",
			legend: "msg.impsigner.crl",
			fields: [
				{ id: "validityDaysCRL", label: "msg.impsigner.validityDaysCRL", type: "number", min:1, max:18263, required: true, editable: editable, value: (this.model.validityDaysCRL ? this.model.validityDaysCRL : 365) },
			]
		}

		this.form.push(crl);
	}

	return this.form;
}



ImportX509SignerRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	var trustCenter = this.getRecipient();

	if (this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_COMPLETE) {
		this.actionList.push("action.save");

		if (trustCenter.isCertificationOfficer(user)) {
			if (this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_EDIT) {
				this.actionList.push("action.impsigner.import");
			}
		}
	}

	return this.actionList;
}



ImportX509SignerRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		GPSystem.log(GPSystem.DEBUG, module.id, "action list :" + actionList);
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.save":
			this.setStatusInfo("Saved");
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
			break;
		case "action.impsigner.import":
			this.importSigner(user);
	}

	this.commit(user.id);
	return true;
}



ImportX509SignerRequestModel.prototype.getPolicy = function() {
	var subspec = this.service.getKeySpecification(this.model.subKeySpecId);

	var content = {
		dnmask: this.model.dnmask,
		validityDaysCertificates: this.model.subCertValidityDays,
		validityDaysCRL: this.model.validityDaysCRL,
		pathLenConstraint: this.model.caPathLen, // check if signer certificate has already a ca path length constraint
 		keySpecification: subspec,
	};

	return content;
}



ImportX509SignerRequestModel.prototype.importSigner = function(user, signerTemplate) {
	GPSystem.log(GPSystem.DEBUG, module.id, "importSigner()");

	var policy = this.getPolicy();

	policy.distinguishedName = signerTemplate.cert.getSubjectDNString();

	var key = signerTemplate.cert.getPublicKey();
	if (key.getComponent(Key.MODULUS)) {
		policy.signatureAlgorithm = Crypto.RSA_SHA256;
	} else {
		policy.signatureAlgorithm = Crypto.ECDSA_SHA256;
	}

	var native = signerTemplate.cert.getNative();
	var pathLen = native.getBasicConstraints();
	if (pathLen != -1) {
		policy.pathLenConstraint = pathLen;
		this.model.caPathLen = pathLen;
	}

	var template = {
		keyDomain: signerTemplate.keyDomain,
		content: JSON.stringify(policy, null, '\t')
	}

	var signerDAO = this.service.daof.getSignerDAO();
	var signerVO = signerDAO.newSigner(signerTemplate.holder, signerTemplate.name, signerTemplate.keyId, template);

	if (signerTemplate.holder.subjectId != this.getRecipientId()) {
		GPSystem.log(GPSystem.DEBUG, module.id, "Reassign subject id from " + signerTemplate.holder.subjectId + " to " + this.getRecipientId());
		var holderDAO = this.service.daof.getHolderDAO();
		holderDAO.updateSubjectId(signerTemplate.holder, this.getRecipientId());
	}

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
	this.setStatusInfo("Signer Imported");
	this.commit(user.id);
}
