/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for enrolling a person
 */

var EnrollBaseRequestController = require('pki-as-a-service/processes/EnrollBaseRequestController').EnrollBaseRequestController;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.enrp.title": "Enroll Person"
	}}
]);



/**
 * CTOR
 *
 * Derived from EnrollBaseRequestController
 */
function EnrollPersonRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new EnrollPersonRequestController()");
}

EnrollPersonRequestController.prototype = Object.create(EnrollBaseRequestController.prototype);
EnrollPersonRequestController.constructor = EnrollPersonRequestController;

exports.EnrollPersonRequestController = EnrollPersonRequestController;



/**
 * Handle POST request to create service request and initiate a GET after POST redirect to
 * the newly created service request.
 *
 * @param {HttpRequest} req the request object
 * @param {HttpResponse} res the response object
 * @param {ServiceRequestController} srct the service request controller in which this controller is embedded.
 * @param {String} action from the POST parameter
 * @type boolean
 * @return true if processing completed, false if it failed
 */
EnrollPersonRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	var ca = this.getTrustCenter(req, res, srct);
	if (!ca.canRead(srct.ui.session.user)) {
		return false;
	}
	if (!ca.isRegistrationOfficer(srct.ui.session.user)) {
		return false;
	}

	var t = {
		process: "EnrollPersonServiceRequest",
		title: srct.i18n("msg.enrp.title"),
		recipientId: ca.getId(),
		content: {
			type: "Person"
		}
	};

	var vo = srct.createServiceRequest(t);

	srct.ui.redirectTo(req, res, req.url, "/sr/" + vo.id);
	return true;
}



/**
 * Handle GET request to show page with information about creating this service request.
 *
 * @param {HttpRequest} req the request object
 * @param {HttpResponse} res the response object
 * @param {ServiceRequestController} srct the service request controller in which this controller is embedded.
 * @type XML
 * @return The XHTML to be embedded in the service request controller frame.
 */
EnrollPersonRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var trustCenter = this.getTrustCenter(req, res, srct);
	var subjectLink = srct.ui.homeURL(req.url) + "/subject/" + trustCenter.getId();

	var form =	<div>
				<h1>Enroll Person</h1>
			</div>

	if (trustCenter.isRegistrationOfficer(srct.ui.session.user)) {
		form.appendChild(
			<div>
				<p>This service request enrolls a new person, managed by the registration role for trust center <a href={subjectLink}>{trustCenter.getName()}</a></p>
				<p>This is different from the process persons use to perform a self-registration. A person with self-registration is not managed by
				   a trust center and the identity is not validated by the registration officer.</p>
				<p>A person is uniquely identified by it's e-mail address, which must be provided in the next step.
				   A name can be assigned, that must allow to uniquely identify the person in the context of the
				   trust center (i.e. given name and surname or a pseudonym). The assigned name is used as common
				   name in certicates issued for this person.</p>
				<p>A token can be registered for the person as part of the service request. During token registration
				   a further service request is created and an activation code is send to the given e-mail address.
				   The person can later use the token and the activation code to register at the portal.</p>
				<p>For an issued token certificates can be requested from one of the certification authorities available for
				   this trust center. Tokens can also be issued at a later stage using the person view.</p>
			</div>
		);

		form.appendChild(
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
				<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		);
	} else {
		form.appendChild(
			<p>You need to be a registration officer of trust center <a href={subjectLink}>{trustCenter.getName()}</a></p>
		)
	}

	return form;
}
