/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview A service request creating a new Scheme Root CA Signer
 */



var CreateCASignerRequestModel		= require('pki-as-a-service/processes/CreateCASignerRequestModel').CreateCASignerRequestModel;
var Holder				= require('scsh/pki-db/Holder').Holder;
var SubjectDAO 				= require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
var PKIXCommon				= require("scsh/x509/PKIXCommon").PKIXCommon;
var DNEncoder				= require("scsh/x509/DNEncoder").DNEncoder;
var X509CertificateIssuer		= require('scsh/x509/X509CertificateIssuer').X509CertificateIssuer;
var PKCS10				= require('scsh/pkcs/PKCS10').PKCS10;
var ServiceRequestModel			= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function CreateX509SignerRequestModel(service, bo) {
	CreateCASignerRequestModel.call(this, service, bo);
}

CreateX509SignerRequestModel.prototype = Object.create(CreateCASignerRequestModel.prototype);
CreateX509SignerRequestModel.constructor = CreateX509SignerRequestModel;

exports.CreateX509SignerRequestModel = CreateX509SignerRequestModel;


CreateX509SignerRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		var editable = (user.id == this.getOriginatorId()) &&
			((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUBMIT));

		var subjectDAO = this.service.daof.getSubjectDAO();
		var ca = subjectDAO.getSubject(this.getRecipientId());

		var ca = this.getRecipient();
		var tokenList = ca.getToken();
		var selectionList = [];
		for (var i = 0; i < tokenList.length; i++) {
			var token = tokenList[i];
			var hsm = this.service.hsmService.getHSMState(token.path);
			if (!hsm || hsm.isOffline() || hsm.error) {
				GPSystem.log(GPSystem.DEBUG, module.id, "unable to get hsm for path " + token.path);
				continue;
			}

			var optgroup = {
				optgroup: true,
				label: token.path,
				value: []
			};

			for (var j = 0; j < hsm.keyDomain.length; j++) {
				var kdid = hsm.keyDomain[j].toString(HEX);

				var isSelected = kdid.equals(this.model.keyDomain);

				if (kdid.length > 16) {
					kdid = kdid.substring(0, 16) + "...";
				}

				optgroup.value.push( { id: j, value: kdid, selected: isSelected } ) ;
			}
			selectionList.push( optgroup );
		}


		// Issuer:
		// Either a self-signed root
		// or a ca of the same subject
		// or a ca from any public subject
		var issuer = [];

		issuer.push( { id: 0, value: "Self-signed Root CA", selected: this.model.issuer == 0 } );

		// ca of the same subject
		var holderDAO = this.service.daof.getHolderDAO();
		var subject = this.getRecipient();
		var ownca = {
			optgroup: true,
			label: subject.getName(),
			value: []
		};

		var caList = holderDAO.getHolderListBySubject(this.getRecipientId());
		ownca.value = this.getCASelectionList(caList);

		if (ownca.value.length > 0) {
			issuer.push(ownca);
		}

		// public ca
		var subjects = subjectDAO.listSubjectsByType(SubjectDAO.TYPE_TRUST_CENTER);
		for (var i = 0; i < subjects.length; i++) {
			var trustcenter = subjects[i];

			if (trustcenter.id == this.getRecipientId()) {
				continue;
			}

			// public? ok
			if (trustcenter.lifecycle == SubjectDAO.LIFECYCLE_PUBLIC) {
				var caList = holderDAO.getHolderListBySubject(trustcenter.id);

				var publicca = {
					optgroup: true,
					label: trustcenter.name,
					value: []
				};

				publicca.value = this.getCASelectionList(caList);

				if (publicca.value.length > 0) {
					issuer.push(publicca);
				}
			}
		}

		this.form = [];

		// Signer cert spec
		// Issuer
		var certSpec = {
			id: "rootcertspec",
			legend: "msg.xsigner.cacertspec",
			fields: [
				{ id: "issuer", label: "msg.xsigner.issuer", type: "select", editable: editable, value: issuer },
			]
		}

		this.form.push(certSpec);

		var subjectData = {
			id: "subjectData",
			legend: "msg.xsigner.cadata",
			fields: [
				{ id: "caName", label: "msg.xsigner.caName", type: "text", size: 50, required: false, editable: false, value: ca.getName() }
			]
		}

		this.form.push(subjectData);

		if (typeof(this.model.issuer) != "undefined") {
			var keyspecs = this.service.getKeySpecificationList();
			var rootkeyspeclist = [];

			if (!this.model.rootKeySpecId) {
				this.model.rootKeySpecId = 1;
				this.model.subKeySpecId = 1;
				this.model.supportedProfiles = [];
			}

			if (this.model.issuer == 0) {
				var maxValidity = 18263;
				var defaultValidityDays = 3650;
				var maxPath = 9;
				var defaultPath = 0;

				var defaultSubCertValidityDays = 730;
				if (defaultSubCertValidityDays > maxValidity) {
					defaultSubCertValidityDays = maxValidity;
				}

				for (var i = 0; i < keyspecs.length; i++) {
					var keyspec = keyspecs[i];

					var e = { id: keyspec.id, value: keyspec.name };
					if (this.model.rootKeySpecId == keyspec.id) {
						e.selected = true;
					}
					rootkeyspeclist.push(e);
				}
			} else  {
				var caModel = this.getCAModel(this.model.issuer);

				var maxValidity = caModel.validityDaysCertificates;
				this.model.rootCertValidityDays = maxValidity;

				var maxPath = caModel.pathLenConstraint ? caModel.pathLenConstraint - 1 : 0;
				var defaultPath = maxPath;
				this.model.caPathLen = maxPath;

				var defaultSubCertValidityDays = 365;
				if (defaultSubCertValidityDays > maxValidity) {
					defaultSubCertValidityDays = maxValidity;
				}

				this.model.rootKeySpecId = caModel.keySpecification.id;
				for (var i = 0; i < keyspecs.length; i++) {
					var keyspec = keyspecs[i];
					if (keyspec.id == caModel.keySpecification.id) {
						var e = { id: keyspec.id, value: keyspec.name, selected: true };
						rootkeyspeclist.push(e);
						break;
					}
				}
			}

			var subkeyspeclist = [];
			for (var i = 0; i < keyspecs.length; i++) {
				var keyspec = keyspecs[i];

				var e = { id: keyspec.id, value: keyspec.name };
				if (this.model.subKeySpecId == keyspec.id) {
					e.selected = true;
				}
				subkeyspeclist.push(e);
			}

			var profiles = this.service.getCertificateProfileList();
			var profilelist = [];

			for (var i = 0; i < profiles.length; i++) {
				var profile = profiles[i];

				var e = { id: profile.id, value: profile.name };
				if (this.model.supportedProfiles.indexOf(profile.id) >= 0) {
					e.selected = true;
				}
				profilelist.push(e);
			}

			// Signer Cert Spec
			certSpec.fields.push(
				{ id: "rootCertValidityDays", label: "msg.xsigner.rootCertValidityDays", type: "number", min: 1, max: maxValidity, required: true, editable: this.model.issuer == 0 ? editable : false, value: (this.model.rootCertValidityDays ? this.model.rootCertValidityDays : defaultValidityDays) },
				{ id: "caPathLen", label: "msg.xsigner.caPathLen", type: "number", min: 0, max: maxPath, required: true, editable: this.model.issuer == 0 ? editable : false, value: (this.model.caPathLen ? this.model.caPathLen : defaultPath) }
			);

			subjectData.fields.push(
				{ id: "dn", label: "msg.xsigner.dn", type: "text", size: 320, required: true, editable: editable, value: (this.model.dn ? this.model.dn : "c=UT,o=ACME,cn=ACME Root CA") }
			);

			var keySpec = {
				id: "keySpec",
				legend: "msg.xsigner.keySpec",
				fields: []
			};

			// Key Domain
			if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT &&
				selectionList.length > 0) {
				keySpec.fields.push({ id: "keyDomain", label: "msg.xsigner.selectDomain", type: "select", editable: editable, value: selectionList });
			} else if (this.model.keyDomain && isNaN(this.model.keyDomain)) {
				var kdid = this.model.keyDomain;
				if (kdid.length > 16) {
					kdid = kdid.substring(0, 16) + "...";
				}

				keySpec.fields.push(
					{ id: "keyDomain", label: "msg.xsigner.selectDomain", type: "text", size: 50, required: false, editable: false, value: kdid }
				);
			}

			// Signer Key Spec

			keySpec.fields.push(
				{ id: "rootKeySpecId", label: "msg.xsigner.rootKeySpecId", type: "select", editable: this.model.issuer == 0 ? editable : false, value: rootkeyspeclist }
			);

			this.form.push(keySpec);

			var subjectSpec = {
				id: "subjectspec",
				legend: "msg.xsigner.subjectspec",
				fields: [
					{ id: "subKeySpecId", label: "msg.xsigner.subKeySpecId", type: "select", editable: editable, value: subkeyspeclist },
					{ id: "subCertValidityDays", label: "msg.xsigner.subCertValidityDays", type: "number", min: 1, max: maxValidity, required: true, editable: editable, value: (this.model.subCertValidityDays ? this.model.subCertValidityDays : defaultSubCertValidityDays) },
					{ id: "dnmask", label: "msg.xsigner.dnmask", type: "text", size: 320, required: true, editable: editable, value: (this.model.dnmask ? this.model.dnmask : "c=${issuer.c},o=${issuer.o},cn=${servicerequest.commonName}") },
					{ id: "supportedProfiles", label: "msg.xsigner.supportedProfiles", type: "select", multiselect: true, editable: editable, value: profilelist }
				]
			}

			this.form.push(subjectSpec);

			var crl = {
				id: "crl",
				legend: "msg.xsigner.crl",
				fields: [
					{ id: "validityDaysCRL", label: "msg.xsigner.validityDaysCRL", type: "number", min:1, max:18263, required: true, editable: editable, value: (this.model.validityDaysCRL ? this.model.validityDaysCRL : 365) },
				]
			}

			this.form.push(crl);
		}

		this.createRevocationForm(user);
	}

	return this.form;
}



CreateX509SignerRequestModel.prototype.createRevocationForm = function(user) {
	if (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_DELIVER) {
		return;
	}

	var trustCenter = this.getTrustCenterSubject();

	if (trustCenter && trustCenter.isRegistrationOfficer(user)) {
		var reasons = [];
		reasons.push( { id: OCSPQuery.REVOKED, value: "Undefined Reason", selected: ( OCSPQuery.REVOKED == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.KEYCOMPROMISE, value: "Key Compromised", selected: ( OCSPQuery.KEYCOMPROMISE == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.CACOMPROMISE, value: "CA Key Compromised", selected: ( OCSPQuery.CACOMPROMISE == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.AFFILIATIONCHANGED, value: "Affiliation Changed", selected: ( OCSPQuery.AFFILIATIONCHANGED == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.SUPERSEDED, value: "New Certificate Issued", selected: ( OCSPQuery.SUPERSEDED == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.CESSATIONOFOPERATION, value: "CA Discontinued Operation", selected: ( OCSPQuery.CESSATIONOFOPERATION == this.model.revocationReason) } );

		if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_SUSPENDED
			&& this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED) {
			reasons.push( { id: OCSPQuery.CERTIFICATEHOLD, value: "Suspend Certificate", selected: true } );
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUSPENDED) {
			reasons.push( { id: OCSPQuery.GOOD, value: "Resume Certificate", selected: true } );
		}

		reasons.push( { id: OCSPQuery.PRIVILEGEWITHDRAWN, value: "Privileges Withdrawn", selected: ( OCSPQuery.PRIVILEGEWITHDRAWN == this.model.revocationReason) } );

		var canRevoke = this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED;
		this.form.push(
		{
			id: "revocation",
			legend: "msg.cr.revocation",
			fields: [
				{ id: "revocationReason", label: "msg.cr.revocationReason", type: "select", editable: canRevoke, value: reasons }
			]
		});
	}
}



CreateX509SignerRequestModel.prototype.getCASelectionList = function(caList) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getCASelectionList()");
	var list = [];

	var holderDAO = this.service.daof.getHolderDAO();
	for (var i = 0; i < caList.length; i++) {
		var ca = caList[i];
		var caModel = this.getCAModel(ca, true);

		if (caModel && caModel.pathLenConstraint > 0) {
			var path = holderDAO.determineHolderPathList(ca);
			var name = "";

			for (var j = 0; j < path.length; j++) {
				var holder = path[j];
				if (j > 0) {
					name += " / ";
				}
				name += holder.name;
			}

			list.push( { id: ca.id, value: name, selected: ca.id == this.model.issuer } ) ;
		}
	}

	return list;
}



CreateX509SignerRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	if (user.id == this.getOriginatorId()) {
		if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT)) {
			this.actionList.push("action.save");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			if ((typeof(this.model.issuer) != "undefined") &&
				(this.model.keyDomain || typeof(this.model.keyDomain) == "number")) {
				this.actionList.push("action.submit");
			}
		}
	}

	var trustcenter = this.getTrustCenterSubject();
	if (trustcenter && trustcenter.isRegistrationOfficer(user)) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) {
			// TODO editing the dn leads to inconsistend data since the signer has been created already
// 			this.actionList.push("action.save");
			this.actionList.push("action.approve");
			this.actionList.push("action.reject");
		}

		if (this.bo.lifecycle > ServiceRequestModel.LIFECYCLE_DELIVER) {
			if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUSPENDED) {
				this.actionList.push("action.cr.updateStatus");
			} else if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED) {
				this.actionList.push("action.cr.revoke");
			}
		}
	}

	if (trustcenter && trustcenter.isCertificationOfficer(user)) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUBMIT) {
			if (this.findTokenWithDomain()) {
				this.actionList.push("action.xsigner.create");
			}
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_PRODUCE) {
			// this.model.isser will have a value of 0 in this path
			var signer = trustcenter.getSigner(this.model.issuer);
			if (this.findTokenWithDomain(trustcenter, signer.keyDomain)) {
				this.actionList.push("action.xsigner.produce.hsmservice");
			}
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_INUSE
			&& !this.model.hasCRL) {
			if (this.findTokenWithDomain()) {
				this.actionList.push("action.xsigner.createcrl");
			}
		}
	}

	return this.actionList;
}




CreateX509SignerRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		GPSystem.log(GPSystem.DEBUG, module.id, "action list :" + actionList);
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.save":
			this.save()
			break;
		case "action.submit":
			this.save()
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_SUBMIT);
			this.setStatusInfo("Request ready for signer creation");
			break;
		case "action.xsigner.create":
			this.createSigner(user);
			break;
		case "action.approve":
			this.toProduction();
			break;
		case "action.reject":
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_REJECTED);
			this.setStatusInfo("Request rejected");
			break;
		case "action.xsigner.produce.hsmservice":
			this.issueCertificate(user);
			break;
		case "action.xsigner.createcrl":
			this.createCRL(user);
			break;
		case "action.cr.updateStatus":
		case "action.cr.revoke":
			this.revokeCertificate();
			break;
	}

	this.commit(user.id);
	return true;
}



/**
 * Callback from forms processing in Form.setFields()
 */
CreateX509SignerRequestModel.prototype.updateModel = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "updateModel(" + fld.id + "," + val + ")");

	var r = this.validateField(fld, val);

	return r;
}



CreateX509SignerRequestModel.prototype.validateField = function(fld, val) {
	switch(fld.id) {
		case "dnmask":
			return this.checkDNMask(fld, val);
		case "dn":
			return this.checkDN(fld, val);
		default:
			var r = this.validateFieldDefault(fld, val);
			if (r) {
				this.model[fld.id] = val;
			}
			return r;
	}
	return true;
}



CreateX509SignerRequestModel.prototype.checkDNMask = function(fld, val) {
	if (!this.validateFieldDefault(fld, val)) {
		return false;
	}

	try {
		var encoder = new DNEncoder(val);
		encoder.setDN("issuer", {});
		encoder.setMap("subject", { any: true });
		encoder.setMap("servicerequest", { any: true });
		encoder.validate();
		this.model[fld.id] = val;
	}
	catch(e) {
		GPSystem.log(GPSystem.ERROR, module.id, e.message);
		fld.message = e.message;
		return false;
	}
	return true;
}



CreateX509SignerRequestModel.prototype.checkDN = function(fld, val) {
	GPSystem.log(GPSystem.DEBUG, module.id, "checkDN(" + val + ")");
	if (!this.validateFieldDefault(fld, val)) {
		return false;
	}

	try {
		var rdn = PKIXCommon.parseDN(val);
		if (rdn.length == 0) {
			fld.message = "RDN can't be empty";
			return false;
		}

		var cn = PKIXCommon.findRDN(rdn, "CN");
		var holderDAO = this.service.daof.getHolderDAO();

		if (this.model.issuer) {
			var caModel = this.getCAModel(this.model.issuer);
			var encoder = new DNEncoder(caModel.dnmask);

			var sr = {
				c: PKIXCommon.findRDN(rdn, "C"),
				o: PKIXCommon.findRDN(rdn, "O"),
				commonName: cn
			};

			encoder.setMap("servicerequest", sr);
			encoder.setDN("issuer", caModel.distinguishedName);

// 			encoder.setMap("subject", subject.getSubjectMap());

			var dn = encoder.encode();
			var dn = PKIXCommon.dnToString(dn);

			if (!encoder.validate()) {
				fld.message = "DN validation fails";
				return false;
			}

			var holder = holderDAO.getHolderByTypeAndName(this.model.issuer, cn, Holder.X509);
		} else {
			var dn = val;
			var holder = holderDAO.getHolder("/" + cn, Holder.X509);
		}

		if (holder) {
			fld.message = "A CA with that name does already exist";
			GPSystem.log(GPSystem.DEBUG, module.id, fld.message);
			return false;
		}
	}
	catch(e) {
		GPSystem.log(GPSystem.ERROR, module.id, e.message + "\n" + e.stack);
		fld.message = e.message;
		return false;
	}
	this.model[fld.id] = dn;
	return true;
}



CreateX509SignerRequestModel.prototype.getCAModel = function(holderOrId, checkRevocationStatus) {
	if (holderOrId instanceof Holder) {
		var holder = holderOrId;
	} else {
		var holderDAO = this.service.daof.getHolderDAO();
		var holder = holderDAO.getHolderById(this.model.issuer);
	}

	var certDAO = this.service.daof.getCertificateDAO();
	var cert = certDAO.getCurrentCertificate(holder);

	if (!cert) { // no active signer
		return null;
	}

	if (checkRevocationStatus && cert.status != 0) {
		GPSystem.log(GPSystem.DEBUG, module.id, "Certificate #" + cert.id + " has been revoked with reason " + cert.status);
		return null;
	}

	var signerDAO = this.service.daof.getSignerDAO();
	var signer = signerDAO.getSignerByKeyId(holder, cert.keyId);

	return signer.getContent();
}



CreateX509SignerRequestModel.prototype.revokeCertificate = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "revokeCertificate( reason " + this.model.revocationReason + " )");
	if (this.model.revocationReason == OCSPQuery.CERTIFICATEHOLD) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_SUSPENDED);
		this.setStatusInfo("Certificate on hold");
	} else if (this.model.revocationReason == OCSPQuery.GOOD) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
		this.setStatusInfo("Certificate resumed");
	} else {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_TERMINATED);
		this.setStatusInfo("Certificate revoked");
	}

	var invalidityDate = new Date();
	var certDAO = this.service.daof.getCertificateDAO();
	for each (var id in this.model.certIdList) {
		certDAO.updateRevocationStatus(id, this.model.revocationReason);
		if (this.model.revocationReason == OCSPQuery.GOOD) {
			certDAO.updateInvalidityDate(id, null);
		} else {
			certDAO.updateInvalidityDate(id, invalidityDate);
		}
		certDAO.updateRevocationDate(id, null);
	}
}



CreateX509SignerRequestModel.prototype.createSigner = function(user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "Create a new CA Signer");

	var result = this.findTokenWithDomain();
	if (result == null) {
		throw new GPError("CreateX509SignerRequestModel", GPError.INVALID_DATA, 0, "There is no hsm with key domain " + this.model.keyDomain + " online");
		GPSystem.log(GPSystem.DEBUG, module.id, "There is no hsm with key domain " + this.model.keyDomain + " online");
		return;
	}

	// Create CA Holder

	this.model.holderId = X509CertificateIssuer.createCertificateIssuer(this.service.daof, this.model.issuer);

	var rdn = PKIXCommon.parseDN(this.model.dn);
	var cn = PKIXCommon.findRDN(rdn, "CN")
	var trustcenter = this.getRecipient();
	var holderDAO = this.service.daof.getHolderDAO();
	var holder = new Holder();
	holder.id = this.model.holderId;
	holderDAO.updateName(this.model.holderId, cn);
	holderDAO.updateSubjectId(holder, trustcenter.getId());

	// Instantiate CA
	var ca = trustcenter.getX509CertificateIssuer(this.model.holderId);

	// Create Signer

	// Setup policy for ca signer

	var keyspec = this.service.getKeySpecification(this.model.rootKeySpecId);

	var dp = new Key();
	if (keyspec.type == "EC") {
		dp.setComponent(Key.ECC_CURVE_OID, new ByteString(keyspec.curve, OID));
	} else {
		dp.setSize(keyspec.keysize);
	}

	var dn = PKIXCommon.parseDN(this.model.dn);

	var policy = {
		distinguishedName: dn,
		signatureAlgorithm: keyspec.sigalg,
		reqSignatureAlgorithm: keyspec.sigalg,
		validityDaysCertificates: this.model.subCertValidityDays,
		validityDaysCRL: this.model.validityDaysCRL,
		pathLenConstraint: this.model.caPathLen,
 		keySpecification: dp
	}

	if (this.model.issuer == 0 && typeof(this.model.rootCertValidityDays) != "undefined") {
		policy.validityDaysSelfSigned = this.model.rootCertValidityDays;
	}

	ca.setPolicy(policy);

	// Persist signer configuration

	var subspec = this.service.getKeySpecification(this.model.subKeySpecId);

	var content = {
		dnmask: this.model.dnmask,
		distinguishedName: this.model.dn,
		validityDaysCertificates: this.model.subCertValidityDays,
		validityDaysCRL: this.model.validityDaysCRL,
		pathLenConstraint: this.model.caPathLen,
 		keySpecification: subspec,
		signatureAlgorithm: keyspec.sigalg
	};

	var template = {
		keyDomain: new ByteString(this.model.keyDomain, HEX),
		content: JSON.stringify(content, null, '\t')
	}

	var keyid = ca.newSigner(null, template);
	this.model.signingKeyId = keyid;

	if (this.model.issuer == 0) {
		var cert = ca.issueSelfSignedCertificate(keyid, this.getId());
		this.model.certIdList = [ cert.id ];

		// Create Empty CRL
		var crldp = trustcenter.getCRLDistributionPoint(ca);
		ca.addCRLDistributionPoint(crldp);
		if (ca.issueCRL()) {
			this.model.hasCRL = true;
		}

		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
		this.setStatusInfo("Signer created");
	} else {
		this.toApproval();
	}
}



CreateX509SignerRequestModel.prototype.getTrustCenterSubject = function() {
	if (this.model.issuer == 0) {
		return this.getRecipient();
	}

	if (!this.model.issuer) {
		return;
	}

	var holderDAO = this.service.daof.getHolderDAO();
	var issuerHolder = holderDAO.getHolderById(this.model.issuer);

	var trustcenter = this.service.getSubject(issuerHolder.subjectId);

	return trustcenter;
}



CreateX509SignerRequestModel.prototype.toApproval = function() {
	var trustcenter = this.getTrustCenterSubject();
	var raRoleId = trustcenter.getRARoleId();
	this.assignToRole(raRoleId);
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_APPROVE);
	this.setStatusInfo("Request submitted to RA for approval");
}



CreateX509SignerRequestModel.prototype.toProduction = function() {
	var trustcenter = this.getTrustCenterSubject();
	var caRoleId = trustcenter.getCARoleId();
	this.assignToRole(caRoleId);
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_PRODUCE);
	this.setStatusInfo("Request submitted to CA for production");
}



CreateX509SignerRequestModel.prototype.issueCertificate = function(user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "issueCertificate");

	var trustcenter = this.getTrustCenterSubject();
	var parentCA = trustcenter.getX509CertificateIssuer(this.model.issuer);

	var holder = new Holder();
	holder.id = this.model.holderId;

	var requestDAO = this.service.daof.getRequestDAO()
	var request = requestDAO.getRequestByKeyId(holder, this.model.signingKeyId);
	var req = new PKCS10(request.bytes);

	var subject = PKIXCommon.parseDN(this.model.dn);
	var extensions = []; // TODO set extensions
	var cert = parentCA.issueCertificate(holder, req.getPublicKey(), subject, extensions, this.getId());
	GPSystem.log(GPSystem.DEBUG, module.id, "new signer cert " + cert);

	this.model.certIdList = [ cert.id ];

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
	this.setStatusInfo("Signer created");
}



CreateX509SignerRequestModel.prototype.createCRL = function(user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "createCRL()");

	// Create Empty CRL

	var trustCenter = this.getRecipient()
	var ca = trustCenter.getX509CertificateIssuer(this.model.holderId);
	var crl = ca.issueCRL();
	if (crl) {
		this.model.hasCRL = true;
		this.setStatusInfo("CRL created");
	}
}
