/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model for issuing a token
 */

var ServiceRequestModel = require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var CVCertificateStore = require('scsh/eac/CVCertificateStore').CVCertificateStore;
var SmartCardHSM = require('scsh/sc-hsm/SmartCardHSM').SmartCardHSM;
var SmartCardHSMInitializer = require('scsh/sc-hsm/SmartCardHSM').SmartCardHSMInitializer;
var HSMKeyStore = require('scsh/sc-hsm/HSMKeyStore').HSMKeyStore;
var CryptoProvider = require('scsh/sc-hsm/CryptoProvider').CryptoProvider;
var SubjectDAO = require('scsh/pki-db/db/SubjectDAO').SubjectDAO;



/**
 * Data model for issuing a token
 *
 * @constructor
 */
function IssueTokenRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
	this.validationResult = [];
}

IssueTokenRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
IssueTokenRequestModel.constructor = IssueTokenRequestModel;

exports.IssueTokenRequestModel = IssueTokenRequestModel;



IssueTokenRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		this.form = [];

		if (this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_COMPLETE) {
			var editable = ((user.id == this.getOriginatorId()) && (this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_EDIT));

			var subject = this.getSubject();
			var fields = [];
			if (subject.name) {
				fields.push({ id: "name", label: "msg.name", type: "text", editable: false, required: false, value: subject.name });
			}

			if (subject.email) {
				fields.push({ id: "email", label: "msg.email", type: "text", editable: false, required: true, value: subject.email });
			}

			this.form.push( {
				id: "subject",
				legend: "msg.subject",
				fields: fields
				}
			);

			var issuerList = [];
			var trustCenter = this.getRecipient();
			var issuerModel = {
				optgroup: true,
				label: trustCenter.getName(),
				value: []
			};
			var caList = trustCenter.getCAList(user);
			for (var i = 0; i < caList.length; i++) {
				var ca = caList[i];

				issuerModel.value.push( { id: ca.id, value: ca.getReadablePath(), selected: ca.id == this.model.issuer } ) ;
			}
			issuerList.push(issuerModel);

			this.form.push( {
				id: "issuer",
				legend: "msg.it.issuer",
				fields: [
					{ id: "issuer", label: "msg.it.ca", type: "select", editable: editable, value: issuerList },
				]
			});

		}

		if (this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_EDIT) {
			this.form.push( {
				id: "token",
				legend: "msg.token",
				fields: [
					{ id: "tokenPath", label: "msg.tokenpath", type: "text", editable: false, required: false, value: (this.model.selectedTokenPath ? this.model.selectedTokenPath : "")},
					{ id: "tokenStatus", label: "msg.status", type: "text", editable: false, required: false, value: (this.model.tokenStatus ? this.model.tokenStatus : "")}
				]}
			);
		}

		if (this.model.derivationKeyLabel) {
			this.form.push(this.getInitializationFieldset());
		}
	}

	return this.form;
}



IssueTokenRequestModel.prototype.getInitializationFieldset = function() {
	if (this.model.derivationKeyLabel) {
		return  {
			id: "initialization",
			legend: "msg.it.init",
			fields: [
				{ id: "derivationKeyLabel", label: "msg.it.derivationKeyLabel", type: "text", editable: false, required: false, value: this.model.derivationKeyLabel},
				{ id: "rrcMode", label: "msg.it.rrcMode", type: "text", editable: false, required: false, value: this.model.rrcMode},
				{ id: "transportMode", label: "msg.it.transportMode", type: "text", editable: false, required: false, value: this.model.transportMode},
				{ id: "pin", label: "msg.it.pin", type: "text", editable: false, required: false, value: this.model.pin},
				{ id: "noOfKeyDomains", label: "msg.it.noOfKeyDomains", type: "number", editable: false, required: false, value: this.model.noOfKeyDomains}
			]
		}
	}
}



IssueTokenRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_TERMINATED) {
		return this.actionList;
	}

	if (user.id == this.getOriginatorId()) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			this.actionList.push("action.save");
			this.actionList.push("action.selectToken");
			if (this.model.tokenStatus && this.model.tokenStatus != "unregistered_token") {
				this.actionList.push("action.deregisterToken");
			} else if (this.model.tokenStatus == "unregistered_token") {
				this.actionList.push("action.registerToken");
			}
		}

		if (this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_VALIDATE) {
			this.actionList.push("action.initialize");
		}

		if (this.model.issuer && (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_VALIDATE)) {
			this.actionList.push("action.reqCert");
		}

		if (this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_VALIDATE) {
			this.actionList.push("action.deregisterToken");
		}
	}

	return this.actionList;
}



IssueTokenRequestModel.prototype.validateField = function(fld, val) {
	switch(fld.id) {
		default:
			return this.validateFieldDefault(fld, val);
	}
	return true;
}



IssueTokenRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	switch(action) {
		case "action.registerToken":
			if (!this.model.selectedTokenPath) {
				return;
			}

			var subject = this.getSubject();
			if (subject.type == SubjectDAO.TYPE_PERSON) {
				this.registerToken(user, subject.email);
				this.model.tokenStatus = "pending_registration";
			} else {
				var tokenDAO = this.service.daof.getTokenDAO();
				var token = tokenDAO.getToken(this.model.selectedTokenPath);
				tokenDAO.updateSubjectId(token, this.model.subjectId);
				tokenDAO.updateServiceRequestId(token, this.getId());
				this.model.tokenStatus = "registered_token";
			}
			this.setStatusInfo("Token registered");
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_VALIDATE);
			break;
		case "action.deregisterToken":
			if (!this.model.selectedTokenPath) {
				return;
			}
			var tokenDAO = this.service.daof.getTokenDAO();
			tokenDAO.deregisterToken(this.model.selectedTokenPath);
			this.service.removeRegistrationTokenAction(this.model.selectedTokenPath);
			this.model.tokenStatus = "unregistered_token";
			this.setStatusInfo("Token deregistered");
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_TERMINATED);
			break;
		case "action.reqCert":
			this.requestCertificate(user);
			break;
	}

	this.commit(user.id);
	return true;
}



IssueTokenRequestModel.prototype.getSubject = function() {
	var subjectDAO = this.service.daof.getSubjectDAO();
	return subjectDAO.getSubject(this.model.subjectId);
}



IssueTokenRequestModel.prototype.registerToken = function(user, email) {
	if (!this.model.selectedTokenPath) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Cannot start RegisterMyTokenServiceRequest without a token path");
	}

	var controller = this.service.getControllerForProcess("RegisterMyTokenServiceRequest");

	var tokenDAO = this.service.daof.getTokenDAO();
	var token = tokenDAO.getToken(this.model.selectedTokenPath);

	var parameter = {
		token: token,
		lang: user.lang,
		originatorId: this.getOriginatorId(),
		raRoleID: this.getRecipient().getRARoleId(),
		email: email,
		enrollmentServiceRequestId: this.bo.id

	};
	var bo = controller.createRequest(this.service, parameter);
	this.model.registerTokenSRID = bo.id;
}



IssueTokenRequestModel.prototype.requestCertificate = function(user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "requestCertificate(" + user.id + ")");

	var controller = this.service.getControllerForProcess("CertificateServiceRequest");

	var tokenDAO = this.service.daof.getTokenDAO();
	var token = tokenDAO.getToken(this.model.selectedTokenPath);

	var subject = this.getSubject();
	var cn = subject.name ? subject.name : subject.email;

	var template = {
		originatorId: this.model.subjectId,
		recipientId: this.getRecipientId(),
		assignedToRole: this.getRecipient().getRARoleId(),
		commonName: cn,
		parent: this.getId(),
		lang: user.lang,
		issuer: this.model.issuer

	}
	var bo = controller.createInitializedRequest(this.service, template);
	this.model.reqCertSRID = bo.id;

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_APPROVE);
}



IssueTokenRequestModel.prototype.checkSelectedToken = function(user, tokenPath) {
	var tokenDAO = this.service.daof.getTokenDAO();

	this.model.selectedTokenPath = tokenPath;
	this.setStatusInfo("Token identified");

	if (user.tokenPath == tokenPath) { // Verify that the selected token is not the user token
		this.model.tokenStatus = "session_token";
	} else {
		var token = tokenDAO.getToken(tokenPath);
		if (!token) {
			token = tokenDAO.newToken(this.model.selectedTokenPath);
		}

		if (token.subjectId) { // Check if token was assigned in the past
			this.model.tokenStatus = "registered_token";
		} else if (this.service.getRegistrationBOFromTokenActions(token)) {
			this.model.tokenStatus = "pending_registration";
		} else {
			this.model.tokenStatus = "unregistered_token";
		}
	}

	this.commit(user.id);
}



IssueTokenRequestModel.prototype.getTokenPath = function() {
	return this.model.selectedTokenPath;
}



IssueTokenRequestModel.prototype.getDerivationKeyLabel = function() {
	return this.model.derivationKeyLabel;
}



IssueTokenRequestModel.prototype.getRRCMode = function() {
	return this.model.rrcMode;
}



IssueTokenRequestModel.prototype.isTransportMode = function() {
	return this.model.transportMode;
}



IssueTokenRequestModel.prototype.getPIN = function() {
	return this.model.pin;
}



IssueTokenRequestModel.prototype.getNoOfKeyDomains = function() {
	return this.model.noOfKeyDomains;
}



IssueTokenRequestModel.prototype.setInitializeData = function(user, initData) {
	GPSystem.log(GPSystem.DEBUG, module.id, "setInitializeData");
	this.model.derivationKeyLabel = initData.derivationKeyLabel;
	this.model.rrcMode = initData.rrcMode;
	this.model.transportMode = initData.transportMode;
	this.model.pin = initData.pin;
	this.model.noOfKeyDomains = initData.noOfKeyDomains;

	this.commit(user.id);
}



IssueTokenRequestModel.prototype.getHSMList = function() {
	var trustCenter = this.getRecipient();
	var tokenList = trustCenter.getToken();

	var hsmList = [];

	for (var i = 0; i < tokenList.length; i++) {
		var token = tokenList[i];
		hsm = this.service.hsmService.getHSMState(token.path);

		if (!hsm || hsm.isOffline()) {
			GPSystem.log(GPSystem.DEBUG, module.id, "Skip HSM " + hsm + " for path " + token.path);
			continue;
		}

		hsmList.push(hsm);
	}

	return hsmList
}



IssueTokenRequestModel.prototype.generateDerivationKey = function(keyDomain, label) {

	var kd = new ByteString(keyDomain, HEX);
	var cp = this.service.hsmService.getCryptoProvider(kd, true);
	cp.generateKey(label, 256);
}


IssueTokenRequestModel.prototype.getDerivationKey = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "getDerivationKey");

	if (!this.model.derivationKeyLabel) {
		GPSystem.log(GPSystem.DEBUG, module.id, "No derivation key defined");
		return;
	}

	var hsmList = this.getHSMList();

	for (var i = 0; i < hsmList.length; i++) {
		var hsm = hsmList[i];
		var keyStore = new HSMKeyStore(hsm.sc);
		if (keyStore.hasKey(this.model.derivationKeyLabel)) {
			var key = keyStore.getKey(this.model.derivationKeyLabel)
			return key;
		}
	}
}



IssueTokenRequestModel.prototype.generateSOPIN = function() {
	var key = this.getDerivationKey();

	var cmac = key.sign(Crypto.AES_CMAC, new ByteString(this.model.selectedTokenPath, ASCII));

	return cmac.bytes(0, 8);
}



IssueTokenRequestModel.prototype.initialize = function(card, chain, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "initialize token");

	var soPIN = this.generateSOPIN();

	var sci = new SmartCardHSMInitializer(card);

	sci.setInitializationCode(soPIN);

	sci.setResetRetryCounterMode(this.model.rrcMode);

	sci.setTransportPINMode(this.model.transportMode);

	sci.setUserPIN(new ByteString(this.model.pin, ASCII));

	sci.setKeyDomains(this.model.noOfKeyDomains);

	var url = ApplicationServer.instance.serverURL + "/rt/paas"
	sci.setProvisioningURL(url);

	sci.initialize();

	this.persistDeviceCertificateChain(chain);

	if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUBMIT) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_VALIDATE);
	}

	this.setStatusInfo("Token initialized");
	this.commit(user.id);

	return true;
}



IssueTokenRequestModel.prototype.persistDeviceCertificateChain = function(chain) {
	GPSystem.log(GPSystem.DEBUG, module.id, "persistDeviceCertificateChain()");

	var cvcStore = new CVCertificateStore(this.service.daof);
	var crypto = new Crypto();
	var unprocessed = cvcStore.insertCertificates(crypto, [ chain.srca, chain.dica, chain.devicecert ], true);

	if (unprocessed.length != 0) {
		GPSystem.log(GPSystem.ERROR, module.id, "Failed to process " + unprocessed);
	}
}



IssueTokenRequestModel.prototype.handleSmartCardHSM = function(sc, chain, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleCardAction for status: " + this.bo.lifecycle);
	switch(this.bo.lifecycle) {
		case ServiceRequestModel.LIFECYCLE_EDIT:
			this.checkSelectedToken(user, chain.path);
			break;
		default:
			GPSystem.log(GPSystem.ERROR, module.id, "Unexpected handleCard in state " + this.bo.lifecycle);
	}
}



IssueTokenRequestModel.prototype.handleCard = function(card, session, reqinfo) {
	var user = session.user;

	var sc = new SmartCardHSM(card);
	var devAutCert = sc.readBinary(SmartCardHSM.C_DevAut);
	var crypto = new Crypto();
	var chain = SmartCardHSM.validateCertificateChain(crypto, devAutCert);
	if (chain == null) {
		this.addMessage("Could not authenticate SmartCard-HSM");
		return;
	}

	sc.openSecureChannel(crypto, chain.publicKey);

	this.handleSmartCardHSM(sc, chain, user);
}



IssueTokenRequestModel.prototype.notify = function(sr) {
	GPSystem.log(GPSystem.DEBUG, module.id, "notify( " + sr.getLifeCycle() + " )");
	if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) {

		if (sr.getId() == this.model.registerTokenSRID) {
			this.model.registerTokenSRLifeCycle = sr.getLifeCycle()
		}

		if (sr.getId() == this.model.reqCertSRID) {
			this.model.reqCertSRLifeCycle = sr.getLifeCycle()
		}

		if (this.model.registerTokenSRID) { // Subject of type Person
			if (this.model.registerTokenSRLifeCycle == ServiceRequestModel.LIFECYCLE_COMPLETE
				&& this.model.reqCertSRLifeCycle >= ServiceRequestModel.LIFECYCLE_INUSE) {
				this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
				this.setStatusInfo("Person enrolled");
				this.model.tokenStatus = "registered_token";
			}
		} else { // Subject of type System
			if (this.model.reqCertSRLifeCycle >= ServiceRequestModel.LIFECYCLE_INUSE) {
				this.setLifeCycle(ServiceRequestModel.LIFECYCLE_COMPLETE);
				this.setStatusInfo("System enrolled");
			}
		}

		this.commit();
	}
}
