/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2020 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for CreateTrustCenterServiceRequest
 */

var CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
var RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;
var CommonUI = require('scsh/srv-cc1/CommonUI').CommonUI;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
	{ lang: "EN", keys: {
		"msg.casubject.title": "Create TrustCenter Subject",

		"msg.casubject.subjectData": "TrustCenter Data",
		"msg.casubject.name": "Unique Name",
		"msg.casubject.token": "Assigned Token",
		"msg.casubject.lc": "Operation Mode",

		"action.cas.addToken": "Add Token",
		"action.cas.addToken.tooltip": "Add the token inserted at the client to the list of assigned token",
		"action.cas.removeToken": "Remove Token",
		"action.cas.removeToken.tooltip": "Remove the token selected in the Assigned Token list from this TrustCenter",
		"action.cas.changeMode": "Change Operation Mode",
		"action.cas.changeMode.tooltip": "Change the operation mode of the TrustCenter",

	}}
]);



function CreateTrustCenterRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new CreateTrustCenterRequestController()");
}

exports.CreateTrustCenterRequestController = CreateTrustCenterRequestController;



CreateTrustCenterRequestController.prototype.createRequest = function(srct, parameter) {

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "CreateTrustCenterServiceRequest",
		title: srct.i18n("msg.casubject.title"),
		originatorId: srct.ui.session.user.id,
		state: "Created",
		content: {
		}
	};

	var bo = srDAO.newServiceRequest(t);
	return bo;
}



CreateTrustCenterRequestController.prototype.postCreateAction = function(req, res, srct, action) {
	if (srct.ui.session.user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
		var parameter = CommonUI.parseQueryString(req.queryString);
		var bo = this.createRequest(srct, parameter);

		srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
		return true;
	}

	return false;
}



CreateTrustCenterRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.submit":
	case "action.save":
	case "action.cas.changeMode":
	case "action.cas.removeToken":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	case "action.cas.addToken":
	case "action.ca.genroot.usertoken":
		srct.ui.wizard = new CardActionWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.wizard.requestPIN = -1;
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



CreateTrustCenterRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var form =	<div>
				<h1>Create a TrustCenter</h1>
				<p>A TrustCenter is an organisation that operates a Registration and Certification
				Authority. It is a certificate issuing party in a Public Key Infrastructure (PKI).</p>
				<p>The Registration Authority is responsible for processing certification requests,
				   which in particular includes the task of identifying the subject to which
				   a certificate is issued. The Certification Authority is responsible for
				   the key management, including the use of signing keys to create certificates.</p>
				<p>A TrustCenter requires an unique name that a user can recognize when requesting
				   a certificate. Ideally the TrustCenter name is the same as the organisation name
				   in certificates issued by a CA operated by the TrustCenter.</p>
				<p>A TrustCenter can be either private or public. A public TrustCenter can be looked
				   up by any user in the PKI-as-a-Service Portal when requesting a certificate.
				   A private TrustCenter is only visible to users involved in the operation of the
				   TrustCenter or to persons enrolled by the TrustCenter.</p>
				<p>When creating a TrustCenter three roles are created and initially assigned to the
				person creating the TrustCenter:</p>
				<ul>
					<li>TrustCenter Manager - responsible for managing HSM and roles.</li>
					<li>Registration Officer - responsible for processing certification requests.</li>
					<li>Certification Officer - responsible for managing keys and issuing certificates.</li>
				</ul>
				<p>As part of this service request you can assign a SmartCard-HSM token
				   to the TrustCenter. Assigning a SmartCard-HSM is important to establish a link between
				   the HSM's identity and the TrustCenter, so that when the SmartCard-HSM is remotely
				   connected, it can be correctly allocated to the entity. Multiple HSMs can be assigned and
				   assigments can be withdrawn by opening the service request again.</p>
			</div>

	if (srct.ui.session.user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
		form.appendChild(
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
					<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		);
	} else {
		form.appendChild(
			<p>You need a subscription to create a new Trust Center.</p>
		)
	}

	return form;
}



CreateTrustCenterRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	return div;
}
