/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2024 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Plugin implementing processes for SAP Certificate Requests
 */

var PKIAPIConnector = require('scsh/srv-cc1/PKIAPIConnector').PKIAPIConnector;
var ServiceRequestAdapter = require('pki-as-a-service/processes/ServiceRequestAdapter').ServiceRequestAdapter;
var CreateGroupRequestController = require('kess/processes/CreateGroupRequestController').CreateGroupRequestController;
var GroupSignerHolderView = require('kess/holder/GroupSignerHolderView').GroupSignerHolderView;
var GroupMemberHolderView = require('kess/holder/GroupMemberHolderView').GroupMemberHolderView;
var KeyEscrowHolderView = require('kess/holder/KeyEscrowHolderView').KeyEscrowHolderView;
var JoinGroupRequestController = require('kess/processes/JoinGroupRequestController').JoinGroupRequestController;
var IssueKeyDomainMembershipRequestController = require('kess/processes/IssueKeyDomainMembershipRequestController').IssueKeyDomainMembershipRequestController;
var CreateEscrowKeyRequestController = require('kess/processes/CreateEscrowKeyRequestController').CreateEscrowKeyRequestController;
var SendKeyToEscrowRequestController = require('kess/processes/SendKeyToEscrowRequestController').SendKeyToEscrowRequestController;
var ReceiveKeyFromEscrowRequestController = require('kess/processes/ReceiveKeyFromEscrowRequestController').ReceiveKeyFromEscrowRequestController;
var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var EnrolledSubject = require('pki-as-a-service/subjects/EnrolledSubject').EnrolledSubject;
var SubjectDAO = require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
var Person = require('pki-as-a-service/subjects/Person').Person;



function KESSPlugin(service, cfg, desc) {
	this.service = service;
	this.priority = desc.priority;
	this.cfg = cfg.kess;
	this.desc = desc;
	this.service.backendRegistry.addService( {
		id: "kess",
		name: "Key Escrow Service",
		apiURL: cfg.kess.apiURL,
		hasHSMService: true });
}

exports.Plugin = KESSPlugin;



KESSPlugin.prototype.getAPI = function() {
	return new PKIAPIConnector(this.cfg);
}



KESSPlugin.prototype.enumerateRunnableProcesses = function(user, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcesses()");
//	processes.push( { label: "Send Key to Escrow", process: "SendKeyToEscrow" } );
//	processes.push( { label: "Receive Key from Escrow", process: "ReceiveKeyFromEscrow" } );
}



KESSPlugin.prototype.enumerateRunnableProcessesForSubject = function(user, subject, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcessesForSubject()");

	var joinGroup = { label: "Join Group", process: "JoinGroup", section: "Key Escrow" };
	if (subject instanceof TrustCenter) {
		if (subject.isCertificationOfficer(user)) {
			processes.push( { label: "Create Group", process: "CreateGroup", section: "Key Escrow" } );
//			processes.push( { label: "Send Key to Escrow", process: "SendKeyToEscrow", section: "Key Escrow" } );
//			processes.push( { label: "Receive Key from Escrow", process: "ReceiveKeyFromEscrow", section: "Key Escrow" } );
		}
		if (subject.isRegistrationOfficer(user)) {
			processes.push(joinGroup);
		}
	}
	if (subject instanceof EnrolledSubject) {
		var trustCenter = subject.getTrustCenter();
		if (trustCenter && trustCenter.isRegistrationOfficer(user)) {
			processes.push(joinGroup);
//			processes.push( { label: "Send Key to Escrow", process: "SendKeyToEscrow", section: "Key Escrow" } );
//			processes.push( { label: "Receive Key from Escrow", process: "ReceiveKeyFromEscrow", section: "Key Escrow" } );
		}
		if (!trustCenter && typeof(user.roles)  != "undefined") {
			var dao = this.service.daof.getSubjectDAO();

			var factory = this.service.pluginRegistry.getFactoryForSubject(SubjectDAO.TYPE_TRUST_CENTER);
			if (!factory) {
				throw new GPError(module.id, GPError.INVALID_DATA, 0, "Can not find factory for subject type " + SubjectDAO.TYPE_TRUST_CENTER);
			}

			var trustCenterList = dao.listSubjectsByFilter( {type: SubjectDAO.TYPE_TRUST_CENTER} );
			for (var i = 0; i < trustCenterList.length; i++) {
				var trustCenter = factory.getSubject(trustCenterList[i]);
				if (trustCenter.isRegistrationOfficer(user)) {
					processes.push(joinGroup);
					break;
				}
			}
		}
	}
}



KESSPlugin.prototype.enumerateRunnableProcessesForHolderType = function(user, subject, type, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcessesForHolderType()");

	if (subject.isManager(user)) {
		if (subject instanceof TrustCenter) {
			if (type == "GroupMember") {
				processes.push( { label: "Send Key to Escrow", process: "SendKeyToEscrow" } );
				processes.push( { label: "Receive Key from Escrow", process: "ReceiveKeyFromEscrow" } );
				processes.push( { label: "Create Escrow Key", process: "CreateEscrowKey" } );
			}
			if (type == "KeyEscrow") {
				processes.push( { label: "Send Key to Escrow", process: "SendKeyToEscrow" } );
				processes.push( { label: "Receive Key from Escrow", process: "ReceiveKeyFromEscrow" } );
			}
		}
		if (subject instanceof Person) {
			if (type == "GroupMember") {
				processes.push( { label: "Send Key to Escrow", process: "SendKeyToEscrow" } );
				processes.push( { label: "Receive Key from Escrow", process: "ReceiveKeyFromEscrow" } );
			}
		}
	}
}



KESSPlugin.prototype.getFactoryForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForProcess(" + process + ")");

	switch(process) {
		case "CreateGroup":
		case "JoinGroup":
		case "IssueKeyDomainMembership":
		case "CreateEscrowKey":
		case "SendKeyToEscrow":
		case "ReceiveKeyFromEscrow":
			return this;
			break;
	}
	return undefined;
}



KESSPlugin.prototype.getFactoryForHolderType = function(type) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForHolderType(" + type + ")");

	switch(type) {
		case "GroupSigner":
		case "GroupMember":
		case "KeyEscrow":
			return this;
			break;
	}
	return undefined;
}



KESSPlugin.prototype.getServiceRequest = function(bo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getServiceRequest(" + bo + ")");

		switch (bo.process) {
		default:
			var sr = new ServiceRequestAdapter(this.service, bo, this);
			break;
	}
	sr.bo = bo;
	sr.id = sr.bo.id;
	sr.service = this.service;
	return sr;
}



KESSPlugin.prototype.getHolderView = function(holder) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getHolderView(" + holder + ")");

	switch (holder.type) {
	case "GroupSigner":
		return new GroupSignerHolderView(this.service, holder);
	case "GroupMember":
		return new GroupMemberHolderView(this.service, holder);
	case "KeyEscrow":
		return new KeyEscrowHolderView(this.service, holder);
	}
}



KESSPlugin.prototype.getControllerForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForProcess(" + process + ")");
	switch(process) {
		case "CreateGroup":
			return new CreateGroupRequestController(this);
		case "JoinGroup":
			return new JoinGroupRequestController(this);
		case "IssueKeyDomainMembership":
			return new IssueKeyDomainMembershipRequestController(this);
		case "CreateEscrowKey":
			return new CreateEscrowKeyRequestController(this);
		case "SendKeyToEscrow":
			return new SendKeyToEscrowRequestController(this);
		case "ReceiveKeyFromEscrow":
			return new ReceiveKeyFromEscrowRequestController(this);

	}
	return undefined;
}



// KESSPlugin.prototype.getFactoryForSubject = function(type) {
// 	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForSubject(" + type + ")");
//
// 	switch(type) {
// 		case "TrustCenter":
// 			return this;
// 			break;
// 	}
// 	return undefined;
// }



// KESSPlugin.prototype.getSubject = function(bo) {
// 	GPSystem.log(GPSystem.DEBUG, module.id, "getSubject(" + bo + ")");
// 	switch (bo.type) {
// 		case "TrustCenter":
// 			var subject = new TrustCenter(this.service, bo);
// 			break;
// 		default:
// 			throw new GPError(module.id, GPError.INVALID_DATA, 0, "Type " + bo.type + " not supported by this factory");
// 	}
// 	return subject;
// }



// KESSPlugin.prototype.getControllerForSubject = function(type) {
// 	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForSubject(" + type + ")");
// 	switch(type) {
// 		case "TrustCenter":
// 			return new TrustCenterController();
// 	}
// 	return undefined;
// }



KESSPlugin.prototype.toString = function() {
	return "KESSPlugin";
}
