/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for sending a key to escrow
 */

var BackendServiceRequestControllerBase = require('pki-as-a-service/ui/BackendServiceRequestControllerBase').BackendServiceRequestControllerBase;
var KESSRequestController = require('kess/processes/KESSRequestController').KESSRequestController;
var I18N = require('scsh/srv-cc1/I18N').I18N;



I18N.addResources([
 	{ lang: "EN", keys: {
		"msg.kess.receivefromescrow": "Receive Key From Escrow Data",
		"msg.kess.keysInEscrow": "Escrowed Keys",
		"action.kess.generateexchangekey": "Generate Exchange Key",
		"action.kess.rewrapforreceiver": "Rewrap Key for Receiver",
		"action.kess.receivefromescrow": "Receive Key From Escrow",
 	}}
 ]);



function ReceiveKeyFromEscrowRequestController(plugin) {
	GPSystem.log(GPSystem.DEBUG, module.id, "new ReceiveKeyFromEscrowRequestController()");
	KESSRequestController.call(this, plugin);
}

exports.ReceiveKeyFromEscrowRequestController = ReceiveKeyFromEscrowRequestController;

ReceiveKeyFromEscrowRequestController.prototype = Object.create(KESSRequestController.prototype);



ReceiveKeyFromEscrowRequestController.prototype.getSelectEscrowForm = function(srct, groupSignerHolderId) {
	var form =
		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<p>Please select a Key Escrow from which you would like to receive keys.</p>
			<select name={ "keyescrow" }/>
			<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
		</form>

	var l = form.select;

	var holderList = this.getIssuerList(srct, "KeyEscrow");
	if (holderList.length == 0) {
		return;
	}

	for each (var holder in holderList) {
		var gsid = holder.getContent().groupSignerHolderId;
		if (groupSignerHolderId == undefined || groupSignerHolderId == gsid) {
			l.appendChild(<option value={holder.id}>{holder.name}</option>);
		}
	}

	return form;
}



ReceiveKeyFromEscrowRequestController.prototype.postCreateAction = function(req, res, srct, action, subject, holder) {
	var recipientId = this.validateIssuer(srct, "KeyEscrow", req.parameters);
	var originatorId = subject != undefined ? subject.getId() : srct.ui.session.user.id;
	// Validity already checked in validateIssuer()
	var keyEscrowHolderId = parseInt(req.parameters.keyescrow);

	var data = { process: "ReceiveKeyFromEscrow",
		originatorId: originatorId,
		recipientId: recipientId,
		content: {
			keyEscrowHolderId: keyEscrowHolderId
		}
	};

	return BackendServiceRequestControllerBase.prototype.postCreateActionViaAPI.call(this, req, res, srct, data);
}



ReceiveKeyFromEscrowRequestController.prototype.getCreateView = function(req, res, srct, subject, holder) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var view =	<div>
				<h1>Receive Key from Escrow</h1>

				<p>Receiving a key from escrow is a two step operation. First the key must be recovered
				   from the database using the escrow key and then it is forwarded to you as the final recipient.</p>

				<p>This service request will create the final recovery key and forward the public key
				   to the escrow agent. The escrow agent then retrieves the key from the database
				   into his SmartCard-HSM and forwards it to you, encrypted with the
				   recovery key created during this request.</p>
			</div>

	if ((holder != null) && (holder.type == "KeyEscrow")) {
		var form =
			<form class="pure-form" action="" method="post" enctype="multipart/form-data">
				<input type="hidden" id="keyescrow" name="keyescrow" value={ holder.id } />
				<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
			</form>
		view.appendChild(form);
		return view;
	}

	if (holder != null) {
		// Limit to escrows that belong to the group signer that matched the group member
		var keyEscrowList = this.getSelectEscrowForm(srct, holder.parentId);
	} else {
		var keyEscrowList = this.getSelectEscrowForm(srct);
	}

	if (keyEscrowList) {
		view.appendChild(keyEscrowList);
	} else {
		view.appendChild(
			<p>There is no available Key Escrow Key.</p>
		);
	}

	return view;
}
