/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2024 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Plugin implementing processes of the X509 CA
 */

var PKIAPIConnector = require('scsh/srv-cc1/PKIAPIConnector').PKIAPIConnector;
var ServiceRequestAdapter = require('pki-as-a-service/processes/ServiceRequestAdapter').ServiceRequestAdapter;
var TrustCenter = require('pki-as-a-service/subjects/TrustCenter').TrustCenter;
var System = require('pki-as-a-service/subjects/System').System;
var BaseSubjectController = require('pki-as-a-service/subjects/BaseSubjectController').BaseSubjectController;
var RoleDAO = require('scsh/pki-db/db/RoleDAO').RoleDAO;

var BaseServiceRequest			= require('scsh/srv-cc1/BaseServiceRequest').BaseServiceRequest;

var ESTRequestController		= require('pki-as-a-service/processes/ESTRequestController').ESTRequestController;
var ESTRequestModel			= require('pki-as-a-service/processes/ESTRequestModel').ESTRequestModel;

var CreateESTServerRequestController	= require('pki-as-a-service/processes/CreateESTServerRequestController').CreateESTServerRequestController;
var CreateESTServerRequestModel		= require('pki-as-a-service/processes/CreateESTServerRequestModel').CreateESTServerRequestModel;

var ImportTrustedCertificatesRequestController = require('pki-as-a-service/processes/ImportTrustedCertificatesRequestController').ImportTrustedCertificatesRequestController;
var ImportTrustedCertificatesRequestModel = require('pki-as-a-service/processes/ImportTrustedCertificatesRequestModel').ImportTrustedCertificatesRequestModel;

var JoinESTServerServiceRequestController = require('pki-as-a-service/processes/JoinESTServerServiceRequestController').JoinESTServerServiceRequestController;
var JoinESTServerServiceRequestModel	= require('pki-as-a-service/processes/JoinESTServerServiceRequestModel').JoinESTServerServiceRequestModel;



function ESTPlugin(service, cfg, desc) {
	this.service = service;
	this.priority = desc.priority;
	this.cfg = cfg;
	this.desc = desc;
}

exports.Plugin = ESTPlugin;



ESTPlugin.prototype.getAPI = function() {
	return new PKIAPIConnector(this.cfg.x509ca);
}



ESTPlugin.prototype.enumerateRunnableProcesses = function(user, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcesses()");
}



ESTPlugin.prototype.enumerateRunnableProcessesForSubject = function(user, subject, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcessesForSubject()");

	if (subject instanceof TrustCenter) {
		if (subject.isManager(user)) {
			if (user.roles.indexOf(RoleDAO.ID_Subscriber) >= 0) {
				processes.push( { label: "Create EST Server", process: "CreateESTServerServiceRequest", section: "EST" } );
			}
		}

		if (subject.isRegistrationOfficer(user)) {
			processes.push( { label: "Import Trusted Certificates", process: "ImportTrustedCertificatesServiceRequest", section: "EST" } );
			processes.push( { label: "Join EST Server", process: "JoinESTServerServiceRequest", section: "EST" } );
		}
	}
}



ESTPlugin.prototype.enumerateRunnableProcessesForHolderType = function(user, subject, type, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcessesForHolderType()");

	/*
	if (subject.isManager(user)) {
		if (subject instanceof TrustCenter) {
			if (type == "GroupSigner") {
				processes.push( { label: "Manage Group", process: "ManageGroup" } );
				processes.push( { label: "Create Escrow Key", process: "CreateEscrowKey" } );
			}
			if (type == "GroupMember") {
				// processes.push( { label: "", process: "" } );
			}
		}
	}
	*/
}



ESTPlugin.prototype.getFactoryForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForProcess(" + process + ")");

	switch(process) {
		case "ESTServiceRequest":
		case "CreateESTServerServiceRequest":
		case "JoinESTServerServiceRequest":
		case "ImportTrustedCertificatesServiceRequest":
			return this;
			break;
	}

	return undefined;
}



/*
ESTPlugin.prototype.getFactoryForHolderType = function(type) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForHolderType(" + type + ")");

	switch(type) {
		case "X509CA":
			return this;
			break;
	}

	return undefined;
}
*/



ESTPlugin.prototype.getServiceRequest = function(bo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getServiceRequest(" + bo + ")");

	switch (bo.process) {
		case "ESTServiceRequest":
			var sr = new ESTRequestModel(this.service, bo);
			break;
		case "CreateESTServerServiceRequest":
			var sr = new CreateESTServerRequestModel(this.service, bo);
			break;
		case "JoinESTServerServiceRequest":
			var sr = new JoinESTServerServiceRequestModel(this.service, bo);
			break;
		case "ImportTrustedCertificatesServiceRequest":
			var sr = new ImportTrustedCertificatesRequestModel(this.service, bo);
			print("### " + sr.canRead( { id: 1 } ));
			break;
		default:
			var sr = BaseServiceRequest.deserialize(bo.content);
			break;
	}

	sr.bo = bo;
	sr.id = sr.bo.id;
	sr.service = this.service;
	return sr;
}



/*
ESTPlugin.prototype.getHolderView = function(holder) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getHolderView(" + holder + ")");

	switch (holder.type) {
	case "X509CA":
		return new X509SignerHolderView(this.service, holder);
	}
}
*/



ESTPlugin.prototype.getControllerForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForProcess(" + process + ")");

	switch(process) {
		case "ESTServiceRequest":
			return new ESTRequestController();
		case "CreateESTServerServiceRequest":
			return new CreateESTServerRequestController();
		case "JoinESTServerServiceRequest":
			return new JoinESTServerServiceRequestController();
		case "ImportTrustedCertificatesServiceRequest":
			return new ImportTrustedCertificatesRequestController();
	}
	return undefined;
}



ESTPlugin.prototype.getFactoryForSubject = function(type) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForSubject(" + type + ")");

	switch(type) {
		case "ESTServer":
			return this;
			break;
	}
	return undefined;
}



ESTPlugin.prototype.getSubject = function(bo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getSubject(" + bo + ")");

	switch (bo.type) {
		case "ESTServer":
			var subject = new System(this.service, bo);
			break;
		default:
			throw new GPError(module.id, GPError.INVALID_DATA, 0, "Type " + bo.type + " not supported by this factory");
	}
	return subject;
}



ESTPlugin.prototype.getControllerForSubject = function(type) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForSubject(" + type + ")");
	switch(type) {
		case "ESTServer":
			return new BaseSubjectController();
	}
	return undefined;
}



ESTPlugin.prototype.toString = function() {
	return "ESTPlugin";
}
