/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 2016 CardContact Systems GmbH
 * |'##> <##'|  Schuelerweg 38, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Plugin implementing processes for SAP Certificate Requests
 */


var BaseServiceRequest			= require('scsh/srv-cc1/BaseServiceRequest').BaseServiceRequest;

var CreateDFNPKIRegistrationAuthorityRequestModel	= require('sapreq/processes/CreateDFNPKIRegistrationAuthorityRequestModel').CreateDFNPKIRegistrationAuthorityRequestModel;
var CreateDFNPKIRegistrationAuthorityRequestController	= require('sapreq/processes/CreateDFNPKIRegistrationAuthorityRequestController').CreateDFNPKIRegistrationAuthorityRequestController;

var DFNCertificateRequestModel		= require('sapreq/processes/DFNCertificateRequestModel').DFNCertificateRequestModel;
var DFNCertificateRequestController	= require('sapreq/processes/DFNCertificateRequestController').DFNCertificateRequestController;

var SAPCertificateRequestModel		= require('sapreq/processes/SAPCertificateRequestModel').SAPCertificateRequestModel;
var SAPCertificateRequestController	= require('sapreq/processes/SAPCertificateRequestController').SAPCertificateRequestController;

var DFNPKI				= require('sapreq/subjects/DFNPKI').DFNPKI;
var DFNPKIController			= require('sapreq/subjects/DFNPKIController').DFNPKIController;

var Person				= require('pki-as-a-service/subjects/Person').Person;



function SAPReqPlugin(service, cfg, desc) {
	this.service = service;
	this.priority = desc.priority;
	this.cfg = cfg;
	this.desc = desc;

	this.service.dfnPKICfg = cfg.dfnPKI;
	this.service.pdfServiceCfg= cfg.pdfService;
	this.service.sapTCList = SAPReqPlugin.tcList;
}

exports.Plugin = SAPReqPlugin;

var fname = GPSystem.mapFilename("etc/trustcenter-list.js", GPSystem.USR);
print("### " + fname);
if (fname) {
	print("Sourcing trustcenter list from " + fname);
	load(fname);
} else {
	SAPReqPlugin.tcList = {
		"default" : {
			university: "*** Unknown ***",
			chargeCode: "1234567890",
			clientNo: "999",
			clientId: "LUIS-T"
		},
		"LUIS Test" : {
			university: "LUIS Test",
			universityShort: "",
			chargeCode: "1234567890",
			clientNo: "999",
			clientId: "LUIS-T"
		},
		"CardContact TrustCenter" : {
			university: "CardContact TrustCenter",
			universityShort: "CardContact",
			chargeCode: "1234567890",
			clientNo: "999",
			clientId: "CC",
			forename: "Andreas",
			surname: "Schwier",
			orgUnit: "Development",
			abbreviation: "Dev",
			additionalAddressInformation: "Add Addr",
			telephoneNumber: "555 5555",
			street: "Schülerweg",
			houseNumber: "38",
			postalCode: "32429",
			city: "Minden"
		}
	};
}



SAPReqPlugin.prototype.enumerateRunnableProcesses = function(user, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcesses()");

	processes.push( { label: "Request DFN Certificate", process: "DFNCertificateServiceRequest" } );
	processes.push( { label: "Request SAP Certificate", process: "SAPCertificateServiceRequest" } );
}



SAPReqPlugin.prototype.enumerateRunnableProcessesForSubject = function(user, subject, processes) {
	GPSystem.log(GPSystem.DEBUG, module.id, "enumerateRunnableProcessesForSubject()");

	if ((subject instanceof Person) && (subject.isManager(user))) {
		processes.push( { label: "Request SAP Certificate", process: "SAPCertificateServiceRequest" } );
	}
}



SAPReqPlugin.prototype.getFactoryForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForProcess(" + process + ")");

	switch(process) {
		case "CreateDFNPKIRegistrationAuthorityServiceRequest":
		case "DFNCertificateServiceRequest":
		case "SAPCertificateServiceRequest":
			return this;
			break;
	}
	return undefined;
}



SAPReqPlugin.prototype.getServiceRequest = function(bo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getServiceRequest(" + bo + ")");
	switch (bo.process) {
		case "CreateDFNPKIRegistrationAuthorityServiceRequest":
			var sr = new CreateDFNPKIRegistrationAuthorityRequestModel(this.service, bo);
			break;
		case "DFNCertificateServiceRequest":
			var sr = new DFNCertificateRequestModel(this.service, bo);
			break;
		case "SAPCertificateServiceRequest":
			var sr = new SAPCertificateRequestModel(this.service, bo);
			break;
		default:
			var sr = BaseServiceRequest.deserialize(bo.content);
			break;
	}
	sr.bo = bo;
	sr.id = sr.bo.id;
	sr.service = this.service;
	return sr;
}



SAPReqPlugin.prototype.getControllerForProcess = function(process) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForProcess(" + process + ")");
	switch(process) {
		case "CreateDFNPKIRegistrationAuthorityServiceRequest":
			return new CreateDFNPKIRegistrationAuthorityRequestController();
		case "DFNCertificateServiceRequest":
			return new DFNCertificateRequestController();
		case "SAPCertificateServiceRequest":
			return new SAPCertificateRequestController();
	}
	return undefined;
}



SAPReqPlugin.prototype.getFactoryForSubject = function(type) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getFactoryForSubject(" + type + ")");

	switch(type) {
		case "DFNPKI":
			return this;
			break;
	}
	return undefined;
}



SAPReqPlugin.prototype.getSubject = function(bo) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getSubject(" + bo + ")");
	switch (bo.type) {
		case "DFNPKI":
			var subject = new DFNPKI(this.service, bo);
			break;
		default:
			throw new GPError(module.id, GPError.INVALID_DATA, 0, "Type " + bo.type + " not supported by this factory");
	}
	return subject;
}



SAPReqPlugin.prototype.getControllerForSubject = function(type) {
	GPSystem.log(GPSystem.DEBUG, module.id, "getControllerForSubject(" + type + ")");
	switch(type) {
		case "DFNPKI":
			return new DFNPKIController();
	}
	return undefined;
}



SAPReqPlugin.prototype.toString = function() {
	return "SAPReqPlugin";
}
