/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model for requesting an SAP certificate
 */

var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var DNEncoder			= require("scsh/x509/DNEncoder").DNEncoder;
var PKIXCommon			= require("scsh/x509/PKIXCommon").PKIXCommon;



var DFNCertificateRequestModel	= require('sapreq/processes/DFNCertificateRequestModel').DFNCertificateRequestModel;


/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function SAPCertificateRequestModel(service, bo) {
	DFNCertificateRequestModel.call(this, service, bo);
}

SAPCertificateRequestModel.prototype = Object.create(DFNCertificateRequestModel.prototype);
SAPCertificateRequestModel.constructor = SAPCertificateRequestModel;

exports.SAPCertificateRequestModel = SAPCertificateRequestModel;



SAPCertificateRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {

		var editable;
		switch(this.bo.lifecycle) {
			case ServiceRequestModel.LIFECYCLE_NEW:
			case ServiceRequestModel.LIFECYCLE_EDIT:
				if ((user.id == this.getOriginatorId()) || this.isManagerOfRequester(user)) {
					editable = true;
				}
				break;
			case ServiceRequestModel.LIFECYCLE_SUBMIT:
				if (this.isAssignedToBasicKeyUser(user)) {
					editable = true;
				}
				break;
			default:
				editable = false;
		}

		if (this.model.email == undefined) {
			this.model.email = this.getOriginator().bo.email;
		}

		this.form = [{
			id: "applicant",
			legend: "msg.sapcr.applicant",
			fields: [
				{ id: "surname", label: "msg.sapcr.surname", type: "text", size: 64, required: true, editable: editable, value: (this.model.surname ? this.model.surname : "") },
				{ id: "forename", label: "msg.sapcr.forename", type: "text", size: 64, required: true, editable: editable, value: (this.model.forename ? this.model.forename : "") },

				{ id: "university", label: "msg.sapcr.university", type: "text", size: 100, required: true, editable: editable, value: (this.model.university ? this.model.university : "") },
				{ id: "orgUnit", label: "msg.dfncr.orgUnit", type: "text", size: 64, required: true, editable: editable, value: (this.model.orgUnit ? this.model.orgUnit : "") },
				{ id: "abbreviation", label: "msg.sapcr.abbreviation", type: "text", size: 64, required: false, editable: editable, value: (this.model.abbreviation ? this.model.abbreviation : "") },

				{ id: "street", label: "msg.sapcr.street", type: "text", size: 64, required: true, editable: editable, value: (this.model.street ? this.model.street : "") },
				{ id: "houseNumber", label: "msg.sapcr.houseNumber", type: "text", size: 64, required: true, editable: editable, value: (this.model.houseNumber ? this.model.houseNumber : "") },
				{ id: "additionalAddressInformation", label: "msg.sapcr.additionalAddressInformation", type: "text", size: 64, required: false, editable: editable, value: (this.model.additionalAddressInformation ? this.model.additionalAddressInformation : "") },
				{ id: "postalCode", label: "msg.sapcr.postalCode", type: "text", size: 64, required: true, editable: editable, value: (this.model.postalCode ? this.model.postalCode : "") },
				{ id: "city", label: "msg.sapcr.city", type: "text", size: 64, required: true, editable: editable, value: (this.model.city ? this.model.city : "") },

				{ id: "email", label: "msg.sapcr.email", type: "text", size: 64, required: true, editable: false, value: this.model.email },

				{ id: "telephoneNumber", label: "msg.sapcr.telephoneNumber", type: "text", size: 64, required: true, editable: editable, value: (this.model.telephoneNumber ? this.model.telephoneNumber : "") }
			]}
		];

		var requestForm = this.getRequestForm(user, editable);
		this.form.push(requestForm);

		this.appendRAForm(user, editable);

		this.appendReviewForm(user);

		this.appendRevocationForm(user);
	}

	return this.form;
}



SAPCertificateRequestModel.prototype.getCommonName= function() {
	// Add the first five digits of the hash over the email
	// to guarantee uniqueness of the CN
	var crypto = new Crypto();
	var hash = crypto.digest(Crypto.MD5, new ByteString(this.model.email, ASCII));
	var hashStr = hash.toString(HEX).substring(0,5);

	return this.model.forename + " " + this.model.surname + " (" + hashStr + ")";
}



SAPCertificateRequestModel.prototype.getSNCName= function(dnval) {
	var rdn = PKIXCommon.parseDN(dnval);
	var map = {};

	for (var i = 0; i < rdn.length; i++) {
		for (var j in rdn[i]) {
			map[j] = rdn[i][j];
		}
	}

	var snc = "p:";
	snc += "CN" + "=" + map["CN"];
	if (map["OU"]) {
		snc += ", OU" + "=" + map["OU"];
	}
	if (map["O"]) {
		snc += ", O" + "=" + map["O"];
	}
	if (map["L"]) {
		snc += ", L" + "=" + map["L"];
	}
	if (map["ST"]) {
		snc += ", SP" + "=" + map["ST"];
	}
	if (map["C"]) {
		snc += ", C" + "=" + map["C"];
	}

	return snc;
}



SAPCertificateRequestModel.prototype.getRequestForm = function(user, editable) {
	var requestForm = DFNCertificateRequestModel.prototype.getRequestForm.call(this, user, editable);
	for (var i = 0; i < requestForm.fields.length; i++) {
		var field = requestForm.fields[i];
		if (field.id == "dn") {
			var snc = this.getSNCName(field.value);
			requestForm.fields.splice( i, 0,
				{ id: "snc", label: "msg.sapcr.snc", type: "text", size: 256, required: false, editable: false, value: snc }
			);
			return requestForm;
		}
	}
	return requestForm;
}



SAPCertificateRequestModel.prototype.transformSubjectDN = function() {

	var sr = {
		commonName: this.getCommonName(),
		orgUnitName: this.model.clientId
	};

	var prefixes = this.getDNPrefixes();
	var prefix = prefixes[this.model.prefix];

	var rdn = PKIXCommon.parseDN(prefix);
	rdn.reverse();
	prefix = PKIXCommon.dnToString(rdn);

	var mask = prefix + ",ou=${servicerequest.orgUnitName},cn=PN - ${servicerequest.commonName}";

	var d = new DNEncoder(mask);
	d.setMap("servicerequest", sr);

	return d.encode();
}



SAPCertificateRequestModel.prototype.updateModel = function(fld, val) {
	switch(fld.id) {
		case "name":
			val = this.replaceUmlauts(val);
			break;
		case "dn":
			return this.validateSubjectDN(fld, val);
	}
	this.model[fld.id] = val;
	return true;
}



SAPCertificateRequestModel.prototype.getActionList = function(user) {
	var actions = DFNCertificateRequestModel.prototype.getActionList.call(this, user);

	if (((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_VALIDATE) && this.isAssignedToBasicKeyUser(user))
		|| ((this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_VALIDATE) && this.isRegistrationOfficer(user))) {
		// Insert action after save
		var si = this.actionList.indexOf("action.save") + 1;	// Will be -1 -> 0 if not found
		this.actionList.splice(si, 0, "action.sapcr.sendpdf");
	}

	return actions;
}
