/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2016 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request controller for CertificateServiceRequest
 */

var CardActionWizard = require('pki-as-a-service/ui/CardActionWizard').CardActionWizard;
var SubjectDAO = require('scsh/pki-db/db/SubjectDAO').SubjectDAO;
var I18N = require('scsh/srv-cc1/I18N').I18N;
var PKIAPIConnector = require('scsh/srv-cc1/PKIAPIConnector').PKIAPIConnector;
var PKIAPICardActionWizard = require('pki-as-a-service/ui/PKIAPICardActionWizard').PKIAPICardActionWizard;



I18N.addResources([
	{ lang: "EN", keys: {
		"action.sapcr.sendpdf": "Send Application PDF",

		"msg.sapcr.title": "Request SAP Certificate",
		"msg.sapcr.applicant": "Certificate Holder Information",
		"msg.sapcr.ra": "Registration Authority",
		"msg.sapcr.raSignerId": "Registration Authority",

		"msg.sapcr.surname": "Surname",
		"msg.sapcr.forename": "Forename",
		"msg.sapcr.university": "University",
		"msg.sapcr.department": "Department",
		"msg.sapcr.abbreviation": "Department abbr.",
		"msg.sapcr.street": "Street",
		"msg.sapcr.houseNumber": "House Number",
		"msg.sapcr.additionalAddressInformation": "Additional Address Information",
		"msg.sapcr.postalCode": "Postal Code",
		"msg.sapcr.city": "City",
		"msg.sapcr.email": "Email",
		"msg.sapcr.telephoneNumber": "Telephone Number",
		"msg.sapcr.pin": "Revocation Password",
		"msg.sapcr.snc": "SNC Name",

		"msg.sapcr.pdfsend": "Application was send by e-mail",
		"msg.sapcr.reqsend": "Certificate request was send to DFN-PKI",
	}},
	{ lang: "DE", keys: {
		"action.sapcr.sendpdf": "Antrags-PDF senden",

		"msg.sapcr.title": "SAP Zertifikatsantrag",
		"msg.sapcr.applicant": "Antragsteller",
		"msg.sapcr.ra": "Registrierungsstelle",
		"msg.sapcr.raSignerId": "Registrierungsstelle",

		"msg.sapcr.surname": "Name",
		"msg.sapcr.forename": "Vorname",
		"msg.sapcr.university": "Universität",
		"msg.sapcr.department": "Abteilung",
		"msg.sapcr.abbreviation": "Abteilungskürzel",
		"msg.sapcr.street": "Straße",
		"msg.sapcr.houseNumber": "Hausnummer",
		"msg.sapcr.additionalAddressInformation": "Zusatz",
		"msg.sapcr.postalCode": "PLZ",
		"msg.sapcr.city": "Stadt",
		"msg.sapcr.email": "Email",
		"msg.sapcr.telephoneNumber": "Telefon",
		"msg.sapcr.pin": "Sperrkennwort",
		"msg.sapcr.snc": "SNC Name",

		"msg.sapcr.pdfsend": "PDF Antrag wurde als e-mail gesendet",
		"msg.sapcr.reqsend": "Zertifikatsrequest an DFN-PKI weitergeleitet",
	}}
]);



function SAPCertificateRequestController() {
	GPSystem.log(GPSystem.DEBUG, module.id, "new SAPCertificateRequestController()");
}

exports.SAPCertificateRequestController = SAPCertificateRequestController;



SAPCertificateRequestController.prototype.createRequest = function(srct, trustCenter, subjectId) {

	var hs = srct.service.sapTCList[trustCenter.getName()];
	if (hs == undefined) {
		hs = srct.service.sapTCList.default;
	}

	var srDAO = srct.service.daof.getServiceRequestDAO();

	var t = {
		process: "SAPCertificateServiceRequest",
		title: srct.i18n("msg.sapcr.title"),
		originatorId: subjectId,
		recipientId: trustCenter.bo.id,
		state: "New request",
		content: {
			pdfTemplate: "sap_req_template",
			pdfFieldKeys: [
				"sr.content.surname",
				"sr.content.forename",
				"sr.content.university",
				"sr.content.universityShort",
				"sr.content.orgUnit",
				"sr.content.abbreviation",
				"sr.content.street",
				"sr.content.houseNumber",
				"sr.content.additionalAddressInformation",
				"sr.content.postalCode",
				"sr.content.city",
				"sr.content.telephoneNumber",
				"sr.content.subjectDN",
				"sr.content.chargeCode",
				"sr.content.clientNo",
				"sr.content.clientId",
				"sr.id",
				"subject.email"
			]
		}
	};

	for (var i in hs) {
		t.content[i] = hs[i];
	}

	return srDAO.newServiceRequest(t);
}


/*
SAPCertificateRequestController.prototype.createInitializedRequest = function(service, template) {
	var srDAO = service.daof.getServiceRequestDAO();

	var t = {
		parentServiceRequestId: template.parent,
		process: "SAPCertificateServiceRequest",
		title: I18N.t(template.lang, "msg.cr.title"),
		details: template.commonName,
		originatorId: template.originatorId,
		recipientId: template.recipientId,
		state: "Submit",
		lifecycle: '3',
		assignedToRole: template.assignedToRole,
		content: {
			commonName: template.commonName,
			role: template.assignedToRole,
		}
	};

	var bo = srDAO.newServiceRequest(t);
	var sr = service.getServiceRequestById(bo.id);

	sr.transformSubjectDN();
	sr.commit();

	return bo;
}
*/


SAPCertificateRequestController.prototype.postCreateAction = function(req, res, srct, action, forSubject) {
	var trustCenterId = parseInt(req.parameters.id);

	GPSystem.log(GPSystem.DEBUG, module.id, "Selected TrustCenter ID:" + trustCenterId);

	var trustCenter = srct.service.getSubject(trustCenterId);

	if (trustCenter == null) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Can't find trust center");
	}

	if (!trustCenter.canRead(srct.ui.session.user) && !trustCenter.isManagerOf(srct.ui.session.user)) {
		GPSystem.log(GPSystem.DEBUG, module.id, "User is neither associated with trustcenter or managed by trustcenter");
		return false;
	}

	var subjectId;
	if (forSubject) {
		subjectId = forSubject.getId();
	} else {
		subjectId = srct.ui.session.user.id;
	}
	var bo = this.createRequest(srct, trustCenter, subjectId);
	srct.ui.redirectTo(req, res, req.url, "/sr/" + bo.id);
	return true;
}



SAPCertificateRequestController.prototype.postProcessAction = function(req, res, srct, sr, action) {

	switch(action) {
	case "action.save":
	case "action.submit":
	case "action.dfncr.reject":
	case "action.dfncr.forward":
	case "action.sendpdf":
		if (!sr.setFields(srct.ui.session.user, req.parameters)) {
			var page = srct.getProcessView(req, res, sr);
			srct.ui.sendPage(req, res, req.url, page);
			return true;
		}
		break;
	}

	// REST call to the DFN Connector
	// - action requestCertificate
	// - action issueCertificate
	switch(action) {
	case "action.cr.reqgen.usertoken":
	case "action.cr.publish.usertoken":
		srct.ui.wizard = new CardActionWizard(srct.ui, sr, "/" + req.url[1] + "/" + req.url[2]);
		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	case "action.cr.requestRevocation":
	case "action.dfncr.forward":
		var cfg = srct.service.dfnPKICfg;
		var data = { user: srct.ui.session.user.id, action: action, args: { fields: req.parameters } };
		var api = new PKIAPIConnector(cfg);
		var rsp = api.postAction(sr.getId(), data);
		if (rsp.message == "msg.completed") {
			srct.ui.setCompletionMessage(srct.ui.i18n("msg.sapcr.reqsend"));
		} else {
			srct.ui.setErrorMessage(srct.ui.i18n(rsp.message), rsp.detail);
		}
		return false;
	case "action.sapcr.sendpdf":
		var cfg = srct.service.pdfServiceCfg;
		var data = { user: srct.ui.session.user.id, action: "action.sendpdf", args: { fields: req.parameters } };
		var api = new PKIAPIConnector(cfg);
		var rsp = api.postAction(sr.getId(), data);
		if (rsp.message == "msg.completed") {
			srct.ui.setCompletionMessage(srct.ui.i18n("msg.sapcr.pdfsend"));
		} else {
			srct.ui.setErrorMessage(srct.ui.i18n(rsp.message), rsp.detail);
		}
		return false;
	case "action.cr.revoke":
	case "action.approve":
		var cfg = srct.service.dfnPKICfg;
		var api = new PKIAPIConnector(cfg);
		var data = { user: srct.ui.session.user.id, action: action, args: { fields: req.parameters } };
		var redirectUrl = "/" + req.url[1] + "/" + req.url[2];
		srct.ui.wizard = new PKIAPICardActionWizard(srct.ui, api, sr.getId(), data, cfg.rtURL, redirectUrl);

		srct.ui.redirectTo(req, res, req.url, "/wizard");
		return true;
	}

	sr.perform(srct.ui.session.user, action);

	return false;
}



SAPCertificateRequestController.prototype.getCreateView = function(req, res, srct) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div =	<div>
				<h1>Request a Certificate</h1>
				<p>This service will request a new certificate from one of the registered
				   certification authorities.</p>
			</div>

	var form =
		<form class="pure-form" action="" method="post" enctype="multipart/form-data">
			<p>Please select a certification authority from which you would like to receive a certificate.</p>
			<select name="id"/>
			<button class="pure-button" type="submit" name="action" value="create">Start Request</button>
		</form>

	var subjectDAO = srct.ui.service.daof.getSubjectDAO();
	var l = form.select;
	var hasEntries = false;
	var trustCenters = subjectDAO.listSubjectsByType(SubjectDAO.TYPE_TRUST_CENTER);
	for each (var subject in trustCenters) {
		var trustCenter = srct.service.getSubject(subject.id);
		if ((trustCenter.canRead(srct.ui.session.user) || trustCenter.isManagerOf(srct.ui.session.user))
			&& (typeof(trustCenter.bo.getContent().dfnpkiRA) != "undefined")) {
			l.appendChild(<option value={subject.id}>{subject.name}</option>);
			hasEntries = true;
		}
	}

	if (hasEntries) {
		div.appendChild(<p>The service request consists of the following steps:</p>);
		div.appendChild(
			<ol>
				<li>You select from which certification authority you would like to receive a certificate</li>
				<li>You provide informations required by the certification authority</li>
				<li>A key pair is generated on your SmartCard-HSM</li>
				<li>A certificate request is generated and forwarded to a registration officer of the CA</li>
				<li>The registration officer either approves or rejects your request</li>
				<li>If the request is approved, then it is forwarded to the certification officer of the CA</li>
				<li>The certification officer will issue the certificate</li>
				<li>In the last step the certificate is stored on your SmartCard-HSM</li>
			</ol>);
		div.appendChild(form);
	} else {
		div.appendChild(<p>There is no accessible certification authority. This is either the case if there are no operational certification authorities or you are not authorized to access a private certification authority.</p>);
	}
	return div;
}



SAPCertificateRequestController.prototype.getProcessView = function(req, res, srct, sr) {
	default xml namespace = "http://www.w3.org/1999/xhtml";

	var div = <div />;

	div.appendChild(srct.renderForm(req, res, sr));

	if (sr.hasCertificates() > 0) {
		div.appendChild( <br/> );
		div.appendChild( <h2>Certificates</h2> );
		div.appendChild(srct.renderCertificateList(req, res, sr));
	}

	return div;
}
