/**
 *  ---------
 * |.##> <##.|  Open Smart Card Development Platform (www.openscdp.org)
 * |#       #|
 * |#       #|  Copyright (c) 1999-2023 CardContact Software & System Consulting
 * |'##> <##'|  Andreas Schwier, 32429 Minden, Germany (www.cardcontact.de)
 *  ---------
 *
 *  This file is part of OpenSCDP.
 *
 *  OpenSCDP is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  OpenSCDP is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with OpenSCDP; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @fileoverview Service request model for requesting a certificate from the DFN-PKI
 */

var ServiceRequestModel		= require('pki-as-a-service/service/ServiceRequestModel').ServiceRequestModel;
var KeyDomain			= require('pki-as-a-service/service/KeyDomain').KeyDomain;
var SmartCardHSM		= require('scsh/sc-hsm/SmartCardHSM').SmartCardHSM;
var HSMKeyStore			= require("scsh/sc-hsm/HSMKeyStore").HSMKeyStore;
var CryptoProvider		= require('scsh/sc-hsm/CryptoProvider').CryptoProvider;
var DNEncoder			= require("scsh/x509/DNEncoder").DNEncoder;
var PKIXCommon			= require("scsh/x509/PKIXCommon").PKIXCommon;
var X509Signer			= require('scsh/x509/X509Signer').X509Signer;
var X509Name			= require('scsh/x509/X509Name').X509Name;
var X509CertificateStore	= require('scsh/x509/X509CertificateStore').X509CertificateStore;
var Holder			= require('scsh/pki-db/Holder').Holder;
var PKCS10 			= require('scsh/pkcs/PKCS10').PKCS10;



/**
 * Data model for requesting a certificate
 *
 *
 * @constructor
 */
function DFNCertificateRequestModel(service, bo) {
	ServiceRequestModel.call(this, service, bo);
}

DFNCertificateRequestModel.prototype = Object.create(ServiceRequestModel.prototype);
DFNCertificateRequestModel.constructor = DFNCertificateRequestModel;

exports.DFNCertificateRequestModel = DFNCertificateRequestModel;



DFNCertificateRequestModel.VERIFICATION_METHOD_POST_IDENT = "Postident (Coupon wird per Mail zugesandt)";
DFNCertificateRequestModel.VERIFICATION_METHOD_BASIS_KEY_USER = "Hochschulintern (Bitte beachten Sie die Regelungen Ihrer Hochschule)";
DFNCertificateRequestModel.VERIFICATION_METHOD_NOT_NEEDED = "Identitätsfeststellung nicht erneut notwendig";
DFNCertificateRequestModel.verificationMethods = [
	DFNCertificateRequestModel.VERIFICATION_METHOD_POST_IDENT,
	DFNCertificateRequestModel.VERIFICATION_METHOD_BASIS_KEY_USER,
	DFNCertificateRequestModel.VERIFICATION_METHOD_NOT_NEEDED
];



DFNCertificateRequestModel.prototype.getForm = function(user) {
	if (this.form == undefined) {
		var editable;
		switch(this.bo.lifecycle) {
			case ServiceRequestModel.LIFECYCLE_NEW:
			case ServiceRequestModel.LIFECYCLE_EDIT:
				if ((user.id == this.getOriginatorId()) || this.isManagerOfRequester(user)) {
					editable = true;
				}
				break;
			case ServiceRequestModel.LIFECYCLE_VALIDATE:
				editable = this.isAssignedToBasicKeyUser(user);
				break;
			default:
				editable = false;
		}

		var sANEmailRequired = false;
		var emailRequired = true;
		this.form = [{
			id: "applicant",
			legend: "msg.dfncr.applicant",
			fields: [
				{ id: "name", label: "msg.dfncr.name", type: "text", size: 64, required: true, editable: editable, value: (this.model.name ? this.model.name : "") },

				{ id: "email", label: "msg.dfncr.email", type: "text", size: 64, required: emailRequired, editable: editable, value: (this.model.email ? this.model.email : this.getOriginator().bo.email) },

				{ id: "orgUnit", label: "msg.dfncr.orgUnit", type: "text", size: 64, required: false, editable: editable, value: (this.model.orgUnit ? this.model.orgUnit : "") }
			]}
		];

		this.form.push({
			id: "subjectAlternativeName",
			legend: "msg.dfncr.subjectAlternativeName",
			fields: [

				{ id: "subjectAlternativeNameEmail", label: "msg.dfncr.email", type: "textarea", size: 500, required: sANEmailRequired, editable: editable, value: (this.model.subjectAlternativeNameEmail ? this.model.subjectAlternativeNameEmail : "") },
				{ id: "subjectAlternativeNameDNS", label: "msg.dfncr.dns", type: "textarea", size: 500, required: false, editable: editable, value: (this.model.subjectAlternativeNameDNS ? this.model.subjectAlternativeNameDNS : "") },
				{ id: "subjectAlternativeNameIP", label: "msg.dfncr.ip", type: "textarea", size: 500, required: false, editable: editable, value: (this.model.subjectAlternativeNameIP ? this.model.subjectAlternativeNameIP : "") },
				{ id: "subjectAlternativeNameURI", label: "msg.dfncr.uri", type: "textarea", size: 500, required: false, editable: editable, value: (this.model.subjectAlternativeNameURI ? this.model.subjectAlternativeNameURI : "") },
				{ id: "subjectAlternativeNameMicrosoftUPN", label: "msg.dfncr.mupn", type: "textarea", size: 500, required: false, editable: editable, value: (this.model.subjectAlternativeNameMicrosoftUPN ? this.model.subjectAlternativeNameMicrosoftUPN : "") }
			]}
		);

		var requestForm = this.getRequestForm(user, editable);

		var roles = this.getCertificateRoles();
		if (roles) {
			var roleList = []
			for (var i = 0; i < roles.length; i++) {
				var role = roles[i];
				roleList.push( { id: i, value: role, selected: ( role == this.model.role ) } );
			}

			requestForm.fields.push( { id: "role", label: "msg.dfncr.role", type: "select", editable: editable, value: roleList } );
		}

		this.form.push(requestForm);

		this.appendRAForm(user, editable);

		this.appendReviewForm(user);

		this.appendRevocationForm(user);
	}

	return this.form;
}



DFNCertificateRequestModel.prototype.getRequestForm = function(user, editable) {
	var requestForm = {
		id: "Request",
		legend: "msg.dfncr.request",
		fields: [
			{ id: "pin", label: "msg.dfncr.pin", type: "text", size: 64, required: true, editable: editable, value: (this.model.pin ? this.model.pin : "") }
	]};

	var prefixes = this.getDNPrefixes();
	if (prefixes) {
		var prefixList = [];
		for (var i = 0; i < prefixes.length; i++) {
			var prefix = prefixes[i];
			prefixList.push( { id: i, value: prefix, selected: ( i == this.model.prefix ) } );
		}

		if (prefixList.length > 0
			&& this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_SUBMIT) { // TODO this.model.prefix will be removed by dfn connector
			requestForm.fields.push( { id: "prefix", label: "msg.dfncr.prefix", type: "select", required: true, editable: editable, value: prefixList } );
		}
	}

	if (this.model.subjectDN) {
		GPSystem.log(GPSystem.DEBUG, module.id, "### DN from this.model.subjectDN: " + this.model.subjectDN);
		var dnval = this.model.subjectDN;
	} else if (this.hasRequest()) {
		var req = this.getRequest();
		var sdn = req.getSubject();
		var xName = new X509Name(sdn)
		var dnval = xName.toString();
		GPSystem.log(GPSystem.DEBUG, module.id, "### DN from request: " + xName.toString());
	} else if (typeof(this.model.prefix) != "undefined") {
		var dnval = PKIXCommon.dnToString(this.transformSubjectDN());
		GPSystem.log(GPSystem.DEBUG, module.id, "### DN from prefix: " + dnval);
	} else {
		GPSystem.log(GPSystem.DEBUG, module.id, "### No DN");
	}

	if (dnval) {
		var editableDN;
		if (this.model.externalApproval) {
			editableDN = (this.isRegistrationOfficer(user) || this.isAssignedToBasicKeyUser(user)) && this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_APPROVE;
		} else {
			editableDN = (this.isRegistrationOfficer(user) || this.isAssignedToBasicKeyUser(user)) && this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_PRODUCE;
		}
		requestForm.fields.push(
			{ id: "dn", label: "msg.cr.dn", type: "text", size: 256, required: false, editable: editableDN, value: dnval }
		);
	}

	if (this.hasRequest()) {
		var req = this.getRequest();
		requestForm.fields.push(
			{ id: "csr", label: "msg.cr.csr", type: "text", size: 100, required: false, editable: false, value: req.toString() }
		);
	}

	return requestForm;
}



DFNCertificateRequestModel.prototype.appendRAForm = function(user, editable) {
	var rAs = [];
	var raList = this.getRegistrationAuthorities();
	if (raList == undefined) {
		return;
	}
	var signerDAO = this.service.daof.getSignerDAO();
	for (var i = 0; i < raList.length; i++) {
		var signerId = raList[i];
		var ra = signerDAO.getSignerByID(signerId);
		GPSystem.log(GPSystem.DEBUG, module.id, "RA: " + ra);

		var desc = ra.getContent().description + " (RA-ID " + ra.getContent().raID + ")";
		rAs.push( { id: signerId, value: desc, selected: ( signerId == this.model.signerId ) } );
	}

	if (rAs.length == 1) {
		rAs[0].selected = true;
	}

	var raForm = {
		id: "ra",
		legend: "msg.dfncr.ra",
		fields: [
			{ id: "signerId", label: "msg.dfncr.signerId", type: "select", editable: editable, value: rAs }
		]
	};

	if (this.bo.process.equals("DFNCertificateServiceRequest")) {
		raForm.fields.push({ id: "externalApproval", label: "msg.dfncr.externalApproval", type: "checkbox", value: "", checked: this.model.externalApproval ? this.model.externalApproval : false, editable: this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_VALIDATE});
	}

	if (this.bo.lifecycle >= ServiceRequestModel.LIFECYCLE_VALIDATE) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_VALIDATE) {
			var editable = this.isAssignedToBasicKeyUser(user);
		} else if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) {
			var editable = this.isAssignedToRegistrationOfficer(user);
		} else {
			var editable = false;
		}

		var list = [];
		for (var i = 0; i < DFNCertificateRequestModel.verificationMethods.length; i++) {
			list.push( { id: i, value: DFNCertificateRequestModel.verificationMethods[i], selected: ( this.model.identityVerification == i) } );
		}

		raForm.fields.push({ id: "identityVerification", label: "msg.dfncr.identityVerification", type: "select", editable: editable, value: list });

		// TODO Add checkbox for the verified state. Set the editable state depending on the selected verification method
	}

	this.form.push(raForm);
}



DFNCertificateRequestModel.prototype.appendReviewForm = function(user) {
	var creatible = (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) || (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_VALIDATE);
	var fields = [];

	if (this.model.reviewRemark || creatible) {
		var editable = creatible && (this.isAssignedToRegistrationOfficer(user) || this.isAssignedToBasicKeyUser(user));
		fields.push(
			{ id: "reviewRemark", label: "msg.cr.reviewRemark", type: "textarea", size: 4096, rows:10, cols:80, editable: editable, value: (this.model.reviewRemark ? this.model.reviewRemark : "") }
		)
	}

	if (this.isRegistrationOfficer(user)) {
		fields.push(
			{ id: "internalReviewRemark", label: "msg.dfncr.internalReviewRemark", type: "textarea", size: 4096, rows:10, cols:80, editable: true, value: (this.model.internalReviewRemark ? this.model.internalReviewRemark : "") }
		)
	}

	if (fields.length > 0) {
		this.form.push(
		{
			id: "approval",
			legend: "msg.cr.approval",
			fields: fields
		});
	}
}



DFNCertificateRequestModel.prototype.appendRevocationForm = function(user) {
	if (this.bo.lifecycle > ServiceRequestModel.LIFECYCLE_DELIVER &&
		this.bo.lifecycle <= ServiceRequestModel.LIFECYCLE_TERMINATED &&
		this.isRegistrationOfficer(user)) {
		var reasons = [];
		reasons.push( { id: OCSPQuery.REVOKED, value: "Undefined Reason", selected: ( OCSPQuery.REVOKED == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.KEYCOMPROMISE, value: "Key Compromised", selected: ( OCSPQuery.KEYCOMPROMISE == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.AFFILIATIONCHANGED, value: "Affiliation Changed", selected: ( OCSPQuery.AFFILIATIONCHANGED == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.SUPERSEDED, value: "New Certificate Issued", selected: ( OCSPQuery.SUPERSEDED == this.model.revocationReason) } );
		reasons.push( { id: OCSPQuery.CESSATIONOFOPERATION, value: "Cessation Of Operation", selected: ( OCSPQuery.CESSATIONOFOPERATION == this.model.revocationReason) } );

// 			reasons.push( { id: 0, value: "Undefined Reason", selected: ( 0 == this.model.revocationReason) } );
// 			reasons.push( { id: 1, value: "Key Compromised", selected: ( 1 == this.model.revocationReason) } );
// 			reasons.push( { id: 3, value: "Affiliation Changed", selected: ( 3 == this.model.revocationReason) } );
// 			reasons.push( { id: 4, value: "New Certificate Issued", selected: ( 4 == this.model.revocationReason) } );
// 			reasons.push( { id: 5, value: "Cessation Of Operation", selected: ( 5 == this.model.revocationReason) } );

		var canRevoke = this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_TERMINATED;
		this.form.push(
		{
			id: "approval",
			legend: "msg.cr.revocation",
			fields: [
				{ id: "revocationReason", label: "msg.cr.revocationReason", type: "select", editable: canRevoke, value: reasons }
			]
		});
	}
}



DFNCertificateRequestModel.prototype.getCertificateRoles = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "getCertificateRoles()");
	var ra = this.getRegistrationAuthority();
	if (!ra) {
		return null;
	}

	var caHolderId = ra.holderId;
	var holderDAO = this.service.daof.getHolderDAO();
	var holder = holderDAO.getHolderById(caHolderId);

	var subjectDAO = this.service.daof.getSubjectDAO();
	var dfnSubject = subjectDAO.getSubject(holder.subjectId);
	var roles = dfnSubject.getContent().roles;
	return roles;
}



DFNCertificateRequestModel.prototype.getDNPrefixes = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "getDNPrefixes()");
	var ra = this.getRegistrationAuthority();
	if (ra) {
		return ra.getContent().dnPrefixes;
	}
}



DFNCertificateRequestModel.prototype.isRegistrationOfficer = function(user) {
	var raRoleId = this.getRARoleId();

	if (raRoleId == -1) {
		return false;	// RA not yet selected
	}

	return (typeof(user.roles) != "undefined") && (user.roles.indexOf(raRoleId) >= 0);
}



DFNCertificateRequestModel.prototype.isAssignedToRegistrationOfficer = function(user) {
	if (!this.bo.assignedToRole) {
		return false;
	}

	var raRoleId = this.getRARoleId();

	if (raRoleId == -1) {
		return false;	// RA not yet selected
	}

	if (this.bo.assignedToRole != raRoleId) {
		return false;
	}

	return (typeof(user.roles) != "undefined") && (user.roles.indexOf(raRoleId) >= 0);
}



DFNCertificateRequestModel.prototype.isAssignedToBasicKeyUser = function(user) {
	if (!this.bo.assignedToRole) {
		return false;
	}

	var recipient = this.getRecipient();
	var roleId = recipient.getRARoleId();

	return recipient.isRegistrationOfficer(user) && (this.bo.assignedToRole == roleId);
}



DFNCertificateRequestModel.prototype.isManagerOfRequester = function(user) {
	return this.getOriginator().isManager(user);
}



DFNCertificateRequestModel.prototype.getActionList = function(user) {
	if (this.actionList != undefined) {
		return this.actionList;
	}

	this.actionList = [];

	if (user.id == this.getOriginatorId() || this.isManagerOfRequester(user)) {
		if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_NEW) ||
			(this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT)) {
			this.actionList.push("action.save");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_EDIT) {
			this.actionList.push("action.submit");
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_SUBMIT) {
			if (!this.model.requestId) {
				if (this.isManagerOfRequester(user)) {
					this.actionList.push("action.save");
				}
				this.actionList.push("action.cr.reqgen.usertoken");
			}
		}

		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_DELIVER) {
			this.actionList.push("action.cr.publish.usertoken");
		}
	}

	if (this.isAssignedToBasicKeyUser(user)) {
		if (this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_VALIDATE) {
			// changing the dn would require the generation of a new request
			this.actionList.push("action.save");
			this.actionList.push("action.dfncr.forward");
			this.actionList.push("action.dfncr.reject");
		}
	}

	if ((this.bo.lifecycle > ServiceRequestModel.LIFECYCLE_DELIVER) && (this.bo.lifecycle < ServiceRequestModel.LIFECYCLE_TERMINATED)) {
		this.appendRevocationActions(user, this.actionList);
	}

	if ((this.bo.lifecycle == ServiceRequestModel.LIFECYCLE_APPROVE) &&
		this.isAssignedToRegistrationOfficer(user)) {
		this.actionList.push("action.save");

		if (this.model.externalApproval) {
			this.actionList.push("action.dfncr.retrieve");
		} else {
			this.actionList.push("action.approve");
		}

		this.actionList.push("action.dfncr.reject");
	}

	if (this.isRegistrationOfficer(user) && (this.actionList.indexOf("action.save") == -1)) {
		this.actionList.splice(0, 0, "action.save");
	}

	return this.actionList;
}



DFNCertificateRequestModel.prototype.appendRevocationActions = function(user, actionList) {
	if (this.hasCertificates()) {
		if (this.isRegistrationOfficer(user)) {
			if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED) {
				actionList.push("action.cr.revoke");
			}
		}
// ToDo: Add action
//		if (user.id == this.getOriginatorId()) {
//			if (this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_SUSPENDED &&
//				this.bo.lifecycle != ServiceRequestModel.LIFECYCLE_TERMINATED) {
//				actionList.push("action.cr.requestRevocation");
//			}
//		}
	}
}



DFNCertificateRequestModel.prototype.getCommonName= function() {
	var crypto = new Crypto();
	var hash = crypto.digest(Crypto.MD5, new ByteString(this.model.email, ASCII));
	var hashStr = hash.toString(HEX).substring(0,5);

	return this.model.name + " (" + hashStr + ")";
}



DFNCertificateRequestModel.prototype.transformSubjectDN = function() {

	var sr = {
		commonName: this.getCommonName(),
	};

	var prefixes = this.getDNPrefixes();
	var prefix = prefixes[this.model.prefix];
	var rdn = PKIXCommon.parseDN(prefix);
	rdn.reverse();
	prefix = PKIXCommon.dnToString(rdn);

	/*
	 * DFN PKI SOAP message:
	 * Die Angabe von GN und SN bzw. Pseudonym oder einem Gruppennamen ist verpflichtend.
	 * Der CN von Pseudonym- oder Gruppenzertifikaten muss mit dem entsprechendem Präfix beginnen.
	 */
	var mask = prefix + ",cn=PN - ${servicerequest.commonName}";
// 	sr.sn = this.model.surname;
// 	sr.gn = this.model.forename;
// 	var mask = prefix + ",sn=${servicerequest.sn},g=${servicerequest.gn},cn=${servicerequest.commonName} (" + hashStr + ")";


	var d = new DNEncoder(mask);
	d.setMap("servicerequest", sr);

	// d.validate();
	return d.encode();
}



DFNCertificateRequestModel.prototype.toBasicKeyUser = function() {
	var raRoleId = this.getRecipient().getRARoleId();
	this.assignToRole(raRoleId);
	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_VALIDATE);
	this.setStatusInfo("Request submitted to Basis-KeyUser for approval");
}



DFNCertificateRequestModel.prototype.getRARoleId = function() {
	var ra = this.getRegistrationAuthority();
	if (!ra) {
		// No RA selected yet
		return -1;
	}
	return ra.getContent().raRoleId;
}



DFNCertificateRequestModel.prototype.validateSubjectDN = function(fld, val) {
	val = this.replaceUmlauts(val);
	try	{
		var subjectDN = PKIXCommon.parseDN(val);
	}
	catch(e) {
		fld.message = e.message;
		return false;
	}
	this.model.subjectDN = val;
	return true;
}



DFNCertificateRequestModel.prototype.replaceUmlauts = function(str) {
	GPSystem.log(GPSystem.DEBUG, module.id, "replaceUmlauts before: " + str);
	str = str.replace(/Ä/g, "Ae");
	str = str.replace(/ä/g, "ae");

	str = str.replace(/Ö/g, "Oe");
	str = str.replace(/ö/g, "oe");

	str = str.replace(/Ü/g, "Ue");
	str = str.replace(/ü/g, "ue");

	str = str.replace(/ß/g, "ss");
	GPSystem.log(GPSystem.DEBUG, module.id, "replaceUmlauts after: " + str);
	return str;
}



DFNCertificateRequestModel.prototype.updateModel = function(fld, val) {
	switch(fld.id) {
		case "name":
			val = this.replaceUmlauts(val);
			break;
		case "dn":
			return this.validateSubjectDN(fld, val);

		case "role":
			var roles = this.getCertificateRoles();
			val = roles[val];
			break;
		case "subjectAlternativeNameEmail":
		case "subjectAlternativeNameDNS":
		case "subjectAlternativeNameIP":
		case "subjectAlternativeNameURI":
		case "subjectAlternativeNameMicrosoftUPN":
			var list = [];
			if (val.length > 0) {
				var lines = val.split("\r\n");
				for (var i = 0; i < lines.length; i++) {
					var line = lines[i];
					if (line.indexOf(",") > 0) {
						list = list.concat(line.split(","));
					} else {
						list.push(line);
					}
				}

			}
			// Sort out empty entries
			val = [];
			for (var i = 0; i < list.length; i++) {
				var e = list[i];
				if (e.length > 0) {
					val.push(e);
				}
			}
			break;
	}
	this.model[fld.id] = val;
	return true;
}



DFNCertificateRequestModel.prototype.handleAction = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleAction(" + user.id + "," + action + ")");

	switch(action) {
		case "action.save":
			this.handleSave();
			break;
		case "action.submit":
			this.updateDetails(this.model.email);
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_SUBMIT);
			this.setStatusInfo("Request being prepared for submission");
			break;
		case "action.dfncr.reject":
			this.assignToRole(0);
			this.setLifeCycle(ServiceRequestModel.LIFECYCLE_REJECTED);
			this.setStatusInfo("Request rejected");
			break;
	}
}



DFNCertificateRequestModel.prototype.perform = function(user, action) {
	GPSystem.log(GPSystem.DEBUG, module.id, "perform(" + user.id + "," + action + ")");

	var actionList = this.getActionList(user);

	if (actionList.indexOf(action) == -1) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Action " + action + " unknown");
	}

	this.handleAction(user, action);

	this.commit(user.id);
	return true;
}



DFNCertificateRequestModel.prototype.handleSave = function() {
	if (this.getLifeCycle() == ServiceRequestModel.LIFECYCLE_NEW) {
		this.setLifeCycle(ServiceRequestModel.LIFECYCLE_EDIT);
	}
	this.setStatusInfo("Request data entered");
}



DFNCertificateRequestModel.prototype.hasRequest = function() {
	return typeof(this.model.requestId) != "undefined";
}



DFNCertificateRequestModel.prototype.getRequest = function() {
	if (!this.request) {
		var requestDAO = this.service.daof.getRequestDAO();
		var req = requestDAO.getRequest(this.model.requestId);
		this.request = new PKCS10(req.bytes);
	}
	return this.request;
}



DFNCertificateRequestModel.prototype.hasCertificates = function() {
	return this.getCertificate() != null;
}



DFNCertificateRequestModel.prototype.getCertificate = function() {
	var certDAO = this.service.daof.getCertificateDAO();
	var certs = certDAO.enumerateCertificatesByServiceRequestId(this.getId());

	var cert = null;
	if (certs.length > 0) {
		cert = certs[0];
	}

	return cert;
}



DFNCertificateRequestModel.prototype.getRegistrationAuthority = function() {
	if (this.ra) {
		GPSystem.log(GPSystem.DEBUG, module.id, "### return cached RA");
		return this.ra;
	}

	if (this.model.signerId) {
		var signerDAO = this.service.daof.getSignerDAO();
		this.ra = signerDAO.getSignerByID(this.model.signerId);
		GPSystem.log(GPSystem.DEBUG, module.id, "### return RA");
		return this.ra;
	}
	GPSystem.log(GPSystem.DEBUG, module.id, "No RA selected");
	return null;
}



DFNCertificateRequestModel.prototype.getRegistrationAuthorities = function() {
	var trustCenter = this.getRecipient();
	var c = trustCenter.bo.getContent();
	var raList = c.dfnpkiRA;
	GPSystem.log(GPSystem.DEBUG, module.id, "dfnpkiRA: " + raList);

	return raList;
}



DFNCertificateRequestModel.prototype.getCertificateHolder = function() {
	var holderDAO = this.service.daof.getHolderDAO();

	var caHolderId = this.getRegistrationAuthority().holderId;

	var certholder = holderDAO.getHolderBySubjectAndParent(this.getOriginatorId(), caHolderId, Holder.X509);

	if (!certholder) {
		var t = {
			name: this.getCommonName(),
			subjectId: this.getOriginatorId()
		}
		var certholder = holderDAO.newHolderForParent(caHolderId, Holder.X509, t);
	}

	return certholder;
}



DFNCertificateRequestModel.prototype.generateKeyPair = function(cpf, deviceCert, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "generateKeyPair()");

	var trustCenter = this.getRecipient();

	// Determine or create holderId under the CA holder
	var certholder = this.getCertificateHolder();

	// TODO Get poliy from DFNPKI subject
// 	var ca = trustCenter.getX509CertificateIssuer(this.model.issuer);
// 	var policy = ca.getSubjectPolicyForRequest();

	var subjectDN = this.model.subjectDN;
	if (subjectDN) {
		subjectDN = PKIXCommon.parseDN(subjectDN);
	} else {
		subjectDN = this.transformSubjectDN();
	}
	GPSystem.log(GPSystem.DEBUG, module.id, "Subject DN = " + JSON.stringify(subjectDN));
	var policy = {}
	policy.distinguishedName = subjectDN;
	policy.requestFormat = "pkcs10";
	policy.overwriteKey = true;

	var dp = new Key();
	dp.setSize(4096);
	policy.keySpecification = dp;
	policy.reqSignatureAlgorithm = Crypto.RSA_SHA256;

	var signer = new X509Signer(this.service.daof, cpf, certholder);
	signer.setPolicy(policy);

	var commonName = this.getCommonName();
	var signerName = trustCenter.getName() + "/" + commonName + " [" + Date() + "]";
	if (signerName.length > 100) {
		signerName = signerName.right(100);
	}

	var defaultId = KeyDomain.getDefaultKeyDomainFromCert(deviceCert);
	var template = {
		keyDomain: defaultId
	}

	var keyid = signer.newSigner(signerName, template);

	var requestDAO = this.service.daof.getRequestDAO();
	var request = requestDAO.getRequestByKeyId(certholder, keyid);
	this.model.requestId = request.id;
	this.model.subjectDN = PKIXCommon.dnToString(subjectDN);

	this.toBasicKeyUser();
}



DFNCertificateRequestModel.prototype.updateSubjectName = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "updateSubjectName()");

	var subject = this.getOriginator();
	if (!subject.bo.name) {
		var name = this.getCommonName();
		GPSystem.log(GPSystem.DEBUG, module.id, "No subject name known, setting " + name);
		var subjectDAO = this.service.daof.getSubjectDAO();
		subjectDAO.updateName(subject.bo, name);
	}
}



DFNCertificateRequestModel.prototype.activateCertificate = function() {
	GPSystem.log(GPSystem.DEBUG, module.id, "activateCertificate()");

	var cert = this.getCertificate();

	var certHolder = new Holder();
	certHolder.id = cert.holderId;

	var holderDAO = this.service.daof.getHolderDAO();
	holderDAO.updateCurrentCertificate(certHolder, cert);
}



DFNCertificateRequestModel.prototype.deliverCertificate = function(sc, deviceCert, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "deliverCertificate()");

	var ks = new HSMKeyStore(sc);
	var cert = this.getCertificate();

	if (!cert) {
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "No certificate with ID " + this.model.certIdList[0] + " found");
	}

	var signerDAO = this.service.daof.getSignerDAO();
	var certHolder = new Holder();
	certHolder.id = cert.holderId;
	var signer = signerDAO.getSignerByKeyId(certHolder, cert.keyId);

	var defaultId = KeyDomain.getDefaultKeyDomainFromCert(deviceCert);

	if (!signer.keyDomain.equals(defaultId)) {
		var expected = signer.keyDomain.bytes(0, 8).toString(HEX) + "...";
		var given = defaultId.bytes(0, 8).toString(HEX) + "...";
		throw new GPError(module.id, GPError.INVALID_DATA, 0, "Wrong token: expected token with key domain " + expected + " but given " + given);
	}

	ks.storeEndEntityCertificate(cert.keyId, cert.bytes);

	this.activateCertificate();

	var cs = new X509CertificateStore(this.service.daof);
	var chain = cs.getCertificateChain(cert.holderId);
	for (var i = 1; i < chain.length - 1; i++) {
		ks.storeCACertificate(chain[i].getSubjectDNString(), chain[i]);
	}

	this.updateSubjectName();

	this.setLifeCycle(ServiceRequestModel.LIFECYCLE_INUSE);
	this.setStatusInfo("Certificate published");
}



DFNCertificateRequestModel.prototype.handleSmartCardHSM = function(sc, chain, user) {
	GPSystem.log(GPSystem.DEBUG, module.id, "handleCardAction for status: " + this.bo.lifecycle);
	switch(this.bo.lifecycle) {
		case ServiceRequestModel.LIFECYCLE_SUBMIT:
			var provider = new CryptoProvider(sc, chain.path);
			var cpf = { getCryptoProvider: function() { return provider }};
			this.generateKeyPair(cpf, chain.devicecert, user);
			break;
		case ServiceRequestModel.LIFECYCLE_DELIVER:
			this.deliverCertificate(sc, chain.devicecert, user);
			break;
		default:
			GPSystem.log(GPSystem.ERROR, module.id, "Unexpected handleCard in state " + this.bo.lifecycle);
			return;
	}

	if (typeof(user) == "undefined") {
		var id = this.getRecipientId();
	} else {
		var id = user.id;
	}
	this.commit(id);
}



DFNCertificateRequestModel.prototype.handleCard = function(card, session, reqinfo) {
	var user = session.user;

	var sc = new SmartCardHSM(card);
	var devAutCert = sc.readBinary(SmartCardHSM.C_DevAut);
	var crypto = new Crypto();
	var chain = SmartCardHSM.validateCertificateChain(crypto, devAutCert);
	if (chain == null) {
		this.setStatusInfo("Could not authenticate SmartCard-HSM");
		return;
	}

	sc.openSecureChannel(crypto, chain.publicKey);

	this.handleSmartCardHSM(sc, chain, user);
}
